#ifndef _dmi_system_
#define _dmi_system_

#include "common.h"

#define DEBUG_MIGRATION FALSE
#define DEBUG_OPTIMIZATION TRUE
#define DEBUG_PACKET FALSE
#define DEBUG_BASIC TRUE
#define DEBUG_TEMPORAL FALSE
#define DEBUG_MEMORY FALSE
#define DEBUG_THREAD FALSE
#define DEBUG_SWEEP FALSE
#define DEBUG_RELAY 0

#if DEBUG_RELAY == 0
#define CONST_LARGE_RELAY_SIZE (16 * 1024)
#define CONST_HIGHLOAD_SIZE (512 * 1024)
#elif DEBUG_RELAY == 1
#define CONST_LARGE_RELAY_SIZE (1ULL * 1024 * 1024 * 1024)
#define CONST_HIGHLOAD_SIZE (1ULL * 1024 * 1024 * 1024)
#elif DEBUG_RELAY == 2
#define CONST_LARGE_RELAY_SIZE 1
#define CONST_HIGHLOAD_SIZE 1
#endif

#define CONST_GMEMORY_USAGE_RATE 0.7
#define CONST_PAGE_WINDOW_SIZE 256
#define CONST_VM_MASK_SIZE 16
#define CONST_VM_MAX (1 << VM_MASK_SIZE)
#define CONST_DEFAULT_STACK_SIZE (4 * 1024 * 1024)
#define CONST_THEAP_SEGMENT_SIZE (1ULL << 44ULL)
#define CONST_PHEAP_SEGMENT_SIZE ((1ULL << 47ULL) - CONST_THEAP_SEGMENT_SIZE)
//#define CONST_PHEAP_SEGMENT_SIZE (1ULL << 40ULL)
#define CONST_FASTSORT_THRESHOLD 12

#define SYS_LIBC_VERSION LIBC_KYUTECH
#define SYS_ID_UNDEF -1
#define SYS_MAIN_THREAD_ID -2
#define SYS_FULL_PAGE_SIZE -1
#define SYS_DMI_NULL -1
#define SYS_DMI_BOOT_ADDR 0
#define SYS_ADDR_BUILD(vm_id, offset) (((int64_t)(vm_id) << (64 - CONST_VM_MASK_SIZE)) + (offset))
#define SYS_ADDR_VM_PART(addr) (int32_t)((addr) >> (64 - CONST_VM_MASK_SIZE))
#define SYS_ADDR_OFFSET_PART(addr) ((addr) & (((int64_t)1UL << (64 - CONST_VM_MASK_SIZE)) - 1))
#define SYS_PAGESIZE_ALIGN_CEIL(addr) (((addr) + PAGESIZE - 1) / PAGESIZE * PAGESIZE)
#define SYS_PAGESIZE_ALIGN_FLOOR(addr) ((addr) - (addr) % PAGESIZE)

#define COMMAND_ADD 20
#define COMMAND_DELETE 21
#define COMMAND_DISABLE 22
#define COMMAND_SYNC 23

#define STATE_CREATE 20
#define STATE_EXIT 21
#define STATE_JOIN 22
#define STATE_DETACH 23
#define STATE_MIGRATE 24

#define STATE_FORBID 20
#define STATE_PERMIT 21

#define STATE_INVALID 20
#define STATE_DOWNVALID 21
#define STATE_UPVALID 22
#define STATE_UNDEF 23

#define TYPE_MASTER 20
#define TYPE_SLAVE 21

#define TYPE_RECV 20
#define TYPE_SEND 21
#define TYPE_HANDLE 22

#define TYPE_SOCKET 20
#define TYPE_PIPE 21

#define LIBC_KYUTECH 20
#define LIBC_KOTOTOI 21

#define ACCESS_ONCE 20
#define ACCESS_INVALIDATE 21
#define ACCESS_UPDATE 22
#define ACCESS_LOCAL 23
#define ACCESS_REMOTE 24
#define ACCESS_ONCE_READ 25
#define ACCESS_INVALIDATE_READ 26
#define ACCESS_UPDATE_READ 27
#define ACCESS_ONCE_GREAD 28
#define ACCESS_INVALIDATE_GREAD 29
#define ACCESS_UPDATE_GREAD 30
#define ACCESS_ONCE_GETSIZE 31
#define ACCESS_INVALIDATE_GETSIZE 32
#define ACCESS_UPDATE_GETSIZE 33
#define ACCESS_LOCAL_WRITE 34
#define ACCESS_REMOTE_WRITE 35
#define ACCESS_LOCAL_GWRITE 36
#define ACCESS_REMOTE_GWRITE 37
#define ACCESS_LOCAL_ATOMIC 38
#define ACCESS_REMOTE_ATOMIC 39
#define ACCESS_LOCAL_SETSIZE 40
#define ACCESS_REMOTE_SETSIZE 41

#define CONTEXT_RUN_THREAD 20
#define CONTEXT_CREATE_THREAD 21
#define CONTEXT_JOIN_THREAD 22
#define CONTEXT_DETACH_THREAD 23
#define CONTEXT_MIGRATE_THREAD 24
#define CONTEXT_WAKE_THREAD 25
#define CONTEXT_JOIN_WORLD 26
#define CONTEXT_LEAVE_WORLD 27
#define CONTEXT_ALLOC_VMS 28
#define CONTEXT_FREE_VMS 29
#define CONTEXT_READ_PAGES 30
#define CONTEXT_GREAD_PAGES 31
#define CONTEXT_EXCLUDE_PAGES 32
#define CONTEXT_GEXCLUDE_PAGES 33
#define CONTEXT_EXCLUDE_PAGE 34
#define CONTEXT_WATCH_PAGE 35
#define CONTEXT_SWEEP_PAGE 36
#define CONTEXT_MIGRATE_PAGE 37
#define CONTEXT_READ_OBJECT 38
#define CONTEXT_GREAD_OBJECT 39
#define CONTEXT_EXCLUDE_OBJECT 40
#define CONTEXT_GEXCLUDE_OBJECT 41
#define CONTEXT_WATCH_OBJECT 42

#define PACKET_REQ_CREATE_THREAD 20
#define PACKET_ACK_CREATE_THREAD 21
#define PACKET_REQ_JOIN_THREAD 22
#define PACKET_ACK_JOIN_THREAD 23
#define PACKET_REQ_DETACH_THREAD 24
#define PACKET_ACK_DETACH_THREAD 25
#define PACKET_REQ_WAKE_THREAD 26
#define PACKET_ACK_WAKE_THREAD 27
#define PACKET_REQ_MIGRATE_THREAD 28
#define PACKET_ACK_MIGRATE_THREAD 29
#define PACKET_REQ_YIELD_THREAD 30
#define PACKET_ACK_YIELD_THREAD 31

#define PACKET_REQ_LOCK_TOKEN 32
#define PACKET_ACK_LOCK_TOKEN 33
#define PACKET_REQ_UNLOCK_TOKEN 34
#define PACKET_ACK_UNLOCK_TOKEN 35
#define PACKET_REQ_DELEGATE_TOKEN 36
#define PACKET_ACK_DELEGATE_TOKEN 37

#define PACKET_REQ_HELLO_WORLD 38
#define PACKET_ACK_HELLO_WORLD 39
#define PACKET_REQ_ALLOC_PEER 40
#define PACKET_ACK_ALLOC_PEER 41
#define PACKET_REQ_PERMIT_PEER 42
#define PACKET_ACK_PERMIT_PEER 43
#define PACKET_REQ_FORBID_PEER 44
#define PACKET_ACK_FORBID_PEER 45
#define PACKET_REQ_FREE_PEER 46
#define PACKET_ACK_FREE_PEER 47
#define PACKET_REQ_ALLOC_VMS 48
#define PACKET_ACK_ALLOC_VMS 49
#define PACKET_REQ_PERMIT_VMS 50
#define PACKET_ACK_PERMIT_VMS 51
#define PACKET_REQ_FORBID_VMS 52
#define PACKET_ACK_FORBID_VMS 53
#define PACKET_REQ_FREE_VMS 54
#define PACKET_ACK_FREE_VMS 55
#define PACKET_REQ_BYE_WORLD 56
#define PACKET_ACK_BYE_WORLD 57

#define PACKET_REQ_READ_PAGE 58
#define PACKET_ACK_READ_PAGE 59
#define PACKET_REQ_RELAY_PAGE 60
#define PACKET_ACK_RELAY_PAGE 61
#define PACKET_REQ_STEAL_PAGE 62
#define PACKET_ACK_STEAL_PAGE 63
#define PACKET_REQ_MIGRATE_PAGE 64
#define PACKET_ACK_MIGRATE_PAGE 65
#define PACKET_REQ_VALIDATE_PAGE 66
#define PACKET_ACK_VALIDATE_PAGE 67
#define PACKET_REQ_INVALIDATE_PAGE 68
#define PACKET_ACK_INVALIDATE_PAGE 69
#define PACKET_REQ_COMMIT_PAGE 70
#define PACKET_ACK_COMMIT_PAGE 71
#define PACKET_REQ_SWEEP_PAGE 72
#define PACKET_ACK_SWEEP_PAGE 73
#define PACKET_REQ_DELEGATE_PAGE 74
#define PACKET_ACK_DELEGATE_PAGE 75


#pragma pack (push, 8)

typedef struct dmi_t
{
  int32_t dmi_id;
  pthread_t recver_pthread;
  pthread_t handler_pthread;
  pthread_t sweeper_pthread;
  struct gate_t *contexts_gate;
  struct gmemory_t *gmemory;
  struct load_t *load;
  struct comm_t *comm;
  struct thread_t *main_thread;
  struct token_t *token;
  xvector_t *peer_gate_xvector;
  xvector_t *vm_gate_xvector;
  xvector_t *thread_gate_xvector;
  taskque_t *packet_taskque;
}dmi_t;

typedef struct dmi_context_t
{
  int32_t dmi_id;
  int32_t context_id;
}dmi_context_t;

typedef struct dmi_thread_t
{
  int32_t dmi_id;
  int32_t thread_id;
}dmi_thread_t;

typedef struct token_t
{
  int8_t owner_flag;
  int8_t lock_flag;
  int32_t owner_id;
  deque_t *dmi_context_deque;
  vector_t *packet_vector;
  pthread_mutex_t mutex;
}token_t;

typedef struct comm_t
{
  int8_t epfd_flag;
  int epfd;
  int listen_sock;
  int self_sock[2];
  int recver_sock[2];
  xvector_t *sock_xvector;
  pthread_cond_t cond;
  pthread_mutex_t mutex;
}comm_t;

typedef struct gmemory_t
{
  int8_t exit_flag;
  int64_t total_size;
  int64_t max_size;
  pthread_cond_t cond;
  pthread_mutex_t mutex;
}gmemory_t;

typedef struct load_t
{
  int64_t size;
  pthread_mutex_t mutex;
}load_t;

typedef struct peer_t
{
  int16_t ip_size;
  uint16_t port;
  char ip[IP_SIZE];
}peer_t;

typedef struct thread_t
{
  int8_t wake_flag;
  int64_t in_size;
  int8_t *in_buf;
  struct context_t *context;
  pthread_t pthread;
  pthread_cond_t cond;
  pthread_mutex_t mutex;
}thread_t;

typedef struct vm_t
{
  int64_t page_size;
  int64_t page_num;
  struct page_t **page_table;
}vm_t;

typedef struct page_t
{
  int8_t state;
  int8_t owner_flag;
  int8_t decor;
  int32_t owner_id;
  int32_t valid_num;
  int32_t stamp;
  int64_t buf_size;
  int64_t slide_size;
  vector_t *account_vector;
  vector_t *context_vector;
  struct order_t *order;
  int8_t *buf;
  pthread_mutex_t mutex;
}page_t;

typedef struct account_t
{
  int8_t state;
  int32_t stamp;
  int32_t seq;
}account_t;

typedef struct order_t
{
  int32_t seq;
  int32_t pending_num;
  vector_t *packet_vector;
}order_t;

typedef struct gate_t
{
  int8_t state;
  int32_t count;
  void *body;
  struct context_t *context;
  void (*func_ptr)(struct context_t *context);
  pthread_mutex_t mutex;
}gate_t;

typedef struct status_t
{
  void *buf;
  struct async_t *async;
}status_t;

typedef struct async_t
{
  int8_t exit_flag;
  int8_t copy_flag;
  int32_t ret;
  int32_t set_num;
  int8_t *src_ptr;
  int64_t *src_offsets;
  int8_t *dst_ptr;
  int64_t *dst_offsets;
  int64_t *copy_sizes;
  pthread_cond_t cond;
  pthread_mutex_t mutex;
}async_t;

typedef struct group_t
{
  int32_t vm_num;
  int32_t total;
  int64_t tls_size;
  int8_t *tls_buf;
  int32_t *vm_ids;
  int32_t *set_nums;
  int64_t *page_ids;
  int64_t *ptr_offsets;
  int64_t *page_offsets;
  int64_t *copy_sizes;
}group_t;

typedef struct range_t
{
  int32_t vm_id;
  int64_t page_id;
  int64_t ptr_offset;
  int64_t page_offset;
  int64_t copy_size;
}range_t;

typedef struct window_t
{
  int32_t size;
  deque_t *status_deque;
  pthread_mutex_t mutex;
}window_t;

typedef struct memory_t
{
  struct theap_t *theap;
  xvector_t *pheap_xvector;
  xidpool_t *xidpool;
  pthread_mutex_t mmap_mutex;
}memory_t;

typedef struct theap_t
{
  int64_t addr;
  pthread_mutex_t mutex;
}theap_t;

typedef struct pheap_t
{
  int64_t stack_size;
  int64_t *dmi_addr_ptr;
  void *stack_ptr;
  list_t *mblock_list;
  list_t *mmapped_list;
  ucontext_t *uc_child_ptr;
  ucontext_t *uc_parent_ptr;
  int64_t dmi_addr;
  int64_t value1;
  int64_t value2;
  int64_t value3;
}pheap_t;

typedef struct mblock_t
{
  void *start_ptr;
  int64_t size;
}mblock_t;

typedef struct mmapped_t
{
  void *start_ptr;
  int64_t size;
  int prot;
  int flags;
}mmapped_t;

typedef struct context_t
{
  int8_t type;
  int8_t master_flag;
  int8_t status_flag;
  int32_t context_id;
  int8_t *buf;
  dmi_t *dmi;
  struct packet_t *recv_packet;
  struct async_t *async;
  pthread_cond_t cond;
  pthread_mutex_t mutex;
  struct
  {
    int8_t access_type;
    int8_t tag;
    int8_t watch_flag;
    int8_t ret_flag;
    int8_t yield_flag;
    int8_t pheap_flag;
    int8_t object_flag;
    int8_t migration_flag;
    int8_t refcount;
    int8_t state;
    int8_t old_state;
    int16_t port;
    int32_t count;
    int32_t total;
    int32_t dmi_id;
    int32_t owner_id;
    int32_t thread_id;
    int32_t new_thread_id;
    int32_t pheap_id;
    int32_t parent_context_id;
    int32_t vm_id;
    int32_t vm_num;
    int32_t set_num;
    int32_t group_num;
    int64_t value1;
    int64_t value2;
    int64_t value3;
    int64_t page_id;
    int64_t page_num;
    int64_t page_size;
    int64_t page_offset;
    int64_t size;
    int64_t slide_size;
    int64_t copy_size;
    int64_t in_size;
    int64_t out_size;
    int64_t stack_size;
    int64_t dmi_addr;
    int sock;
    char *ip; 
    int8_t *in_ptr;
    int8_t *out_ptr;
    int8_t *flag_ptr;
    int32_t *vm_ids;
    int32_t *set_nums;
    int64_t *slide_size_ptr;
    int64_t *addr_offsets;
    int64_t *ptr_offsets;
    int64_t *page_offsets;
    int64_t *sizes;
    int64_t *copy_sizes;
    int64_t *dmi_addr_ptr;
    int64_t *page_sizes;
    int64_t *page_nums;
    int64_t *dmi_addrs;
    int64_t *page_ids;
    struct dmi_thread_t *dmi_thread_ptr;
    struct dmi_thread_t dmi_thread;
    struct dmi_context_t parent_dmi_context;
    struct dmi_context_t join_dmi_context;
  }store;
}context_t;

#define context_access_type(context) (context->store.access_type)
#define context_tag(context) (context->store.tag)
#define context_watch_flag(context) (context->store.watch_flag)
#define context_ret_flag(context) (context->store.ret_flag)
#define context_yield_flag(context) (context->store.yield_flag)
#define context_pheap_flag(context) (context->store.pheap_flag)
#define context_object_flag(context) (context->store.object_flag)
#define context_migration_flag(context) (context->store.migration_flag)
#define context_refcount(context) (context->store.refcount)
#define context_state(context) (context->store.state)
#define context_old_state(context) (context->store.old_state)
#define context_port(context) (context->store.port)
#define context_count(context) (context->store.count)
#define context_total(context) (context->store.total)
#define context_dmi_id(context) (context->store.dmi_id)
#define context_owner_id(context) (context->store.owner_id)
#define context_thread_id(context) (context->store.thread_id)
#define context_new_thread_id(context) (context->store.new_thread_id)
#define context_pheap_id(context) (context->store.pheap_id)
#define context_parent_context_id(context) (context->store.parent_context_id)
#define context_vm_id(context) (context->store.vm_id)
#define context_vm_num(context) (context->store.vm_num)
#define context_set_num(context) (context->store.set_num)
#define context_group_num(context) (context->store.group_num)
#define context_value1(context) (context->store.value1)
#define context_value2(context) (context->store.value2)
#define context_value3(context) (context->store.value3)
#define context_page_id(context) (context->store.page_id)
#define context_page_num(context) (context->store.page_num)
#define context_page_size(context) (context->store.page_size)
#define context_page_offset(context) (context->store.page_offset)
#define context_size(context) (context->store.size)
#define context_slide_size(context) (context->store.slide_size)
#define context_copy_size(context) (context->store.copy_size)
#define context_in_size(context) (context->store.in_size)
#define context_out_size(context) (context->store.out_size)
#define context_stack_size(context) (context->store.stack_size)
#define context_dmi_addr(context) (context->store.dmi_addr)
#define context_sock(context) (context->store.sock)
#define context_ip(context) (context->store.ip)
#define context_in_ptr(context) (context->store.in_ptr)
#define context_out_ptr(context) (context->store.out_ptr)
#define context_flag_ptr(context) (context->store.flag_ptr)
#define context_vm_ids(context) (context->store.vm_ids)
#define context_set_nums(context) (context->store.set_nums)
#define context_slide_size_ptr(context) (context->store.slide_size_ptr)
#define context_addr_offsets(context) (context->store.addr_offsets)
#define context_ptr_offsets(context) (context->store.ptr_offsets)
#define context_page_offsets(context) (context->store.page_offsets)
#define context_sizes(context) (context->store.sizes)
#define context_copy_sizes(context) (context->store.copy_sizes)
#define context_dmi_addr_ptr(context) (context->store.dmi_addr_ptr)
#define context_page_sizes(context) (context->store.page_sizes)
#define context_page_nums(context) (context->store.page_nums)
#define context_dmi_addrs(context) (context->store.dmi_addrs)
#define context_page_ids(context) (context->store.page_ids)
#define context_dmi_thread_ptr(context) (context->store.dmi_thread_ptr)
#define context_dmi_thread(context) (context->store.dmi_thread)
#define context_parent_dmi_context(context) (context->store.parent_dmi_context)
#define context_join_dmi_context(context) (context->store.join_dmi_context)

typedef struct packet_t
{
  int8_t type;
  int8_t delay_flag;
  int32_t src_dmi_id;
  int32_t via_dmi_id;
  int32_t dst_dmi_id;
  int32_t context_id;
  int sock;
  int64_t load_size;
  int64_t page_payload_size;
  int64_t data_payload_size;
  int8_t *page_payload_ptr;
  int8_t *page_payload_buf;
  int8_t *data_payload_ptr;
  int8_t *data_payload_buf;
  struct
  {
    int8_t r8a;
    int8_t r8b;
    int8_t r8c;
    int8_t r8d;
    int16_t r16a;
    int32_t r32a;
    int32_t r32b;
    int32_t r32c;
    int32_t r32d;
    int32_t r32e;
    int64_t r64a;
    int64_t r64b;
    int64_t r64c;
    int64_t r64d;
    int64_t r64e;
    int64_t r64f;
  }reg;
}packet_t;

#define packet_lock_flag(packet) (packet->reg.r8a)
#define packet_access_type(packet) (packet->reg.r8a)
#define packet_ack_flag(packet) (packet->reg.r8a)
#define packet_state(packet) (packet->reg.r8b)
#define packet_tag(packet) (packet->reg.r8b)
#define packet_object_flag(packet) (packet->reg.r8b)
#define packet_pheap_flag(packet) (packet->reg.r8b)
#define packet_ret_flag(packet) (packet->reg.r8c)
#define packet_migrate_flag(packet) (packet->reg.r8d)
#define packet_port(packet) (packet->reg.r16a)
#define packet_relay_flag(packet) (packet->reg.r16a)
#define packet_thread_id(packet) (packet->reg.r32a)
#define packet_owner_id(packet) (packet->reg.r32a)
#define packet_valid_num(packet) (packet->reg.r32a)
#define packet_value(packet) (packet->reg.r32a)
#define packet_context_id(packet) (packet->reg.r32a)
#define packet_dmi_id(packet) (packet->reg.r32b)
#define packet_vm_id(packet) (packet->reg.r32b)
#define packet_set_num(packet) (packet->reg.r32c)
#define packet_stamp(packet) (packet->reg.r32c)
#define packet_vm_num(packet) (packet->reg.r32c)
#define packet_seq2(packet) (packet->reg.r32d)
#define packet_seq(packet) (packet->reg.r32e)
#define packet_page_size(packet) (packet->reg.r64a)
#define packet_page_id(packet) (packet->reg.r64a)
#define packet_page_num(packet) (packet->reg.r64b)
#define packet_page_offset(packet) (packet->reg.r64b)
#define packet_dmi_addr(packet) (packet->reg.r64b)
#define packet_value1(packet) (packet->reg.r64c)
#define packet_out_size(packet) (packet->reg.r64c)
#define packet_value2(packet) (packet->reg.r64d)
#define packet_slide_size(packet) (packet->reg.r64d)
#define packet_value3(packet) (packet->reg.r64e)
#define packet_in_size(packet) (packet->reg.r64e)
#define packet_copy_size(packet) (packet->reg.r64f)
#define packet_stack_size(packet) (packet->reg.r64f)

#pragma pack (pop)

extern __thread int32_t _tls_pheap_id;
extern __thread int32_t _tls_thread_id;
extern int32_t _flag0;
extern int32_t _flag1;
extern int32_t _flag2;
extern int32_t _flag3;
extern double _t0;
extern double _t1;
extern double _t2;
extern double _t3;
extern int64_t dmi_thread(int64_t dmi_addr, int64_t value1, int64_t value2, int64_t value3);
extern void dmi_function(void *page_ptr, int64_t size, void *out_ptr, int64_t out_size, void *in_ptr, int64_t in_size, int8_t tag);

#define DMI_NULL SYS_DMI_NULL
#define DMI_BOOT_ADDR SYS_DMI_BOOT_ADDR

extern void* (*mid_syscall_mmap)(void *start, size_t length, int prot, int flags, int fd, off_t offset);
extern void* (*true_syscall_mmap)(void *start, size_t length, int prot, int flags, int fd, off_t offset);
extern void* (*mid_syscall_mremap)(void *old_address, size_t old_size, size_t new_size, int flags);
extern void* (*true_syscall_mremap)(void *old_address, size_t old_size, size_t new_size, int flags);
extern int (*mid_syscall_munmap)(void *start, size_t length);
extern int (*true_syscall_munmap)(void *start, size_t length);
extern int (*mid_syscall_mprotect)(void *addr, size_t len, int prot);
extern int (*true_syscall_mprotect)(void *addr, size_t len, int prot);
extern int (*mid_syscall_brk)(void *end_data_segment);
extern int (*true_syscall_brk)(void *end_data_segment);


/* spi */

dmi_t* spi_alloc_dmi(uint16_t listen_port, int64_t gmemory_size);
void spi_free_dmi(dmi_t *dmi);
int32_t spi_rank_dmi(dmi_t *dmi, int32_t *dmi_id_ptr);
void spi_join_world(dmi_t *dmi, char *ip, uint16_t port, status_t *status);
void spi_leave_world(dmi_t *dmi, status_t *status);
void spi_create_thread(dmi_t *dmi, dmi_thread_t *dmi_thread_ptr, int32_t dmi_id, int64_t dmi_addr, int64_t stack_size, int8_t pheap_flag, int64_t value1, int64_t value2, int64_t value3, status_t *status);
void spi_join_thread(dmi_t *dmi, dmi_thread_t dmi_thread, int64_t *dmi_addr_ptr, status_t *status);
void spi_detach_thread(dmi_t *dmi, dmi_thread_t dmi_thread, status_t *status);
void spi_migrate_thread(dmi_t *dmi, dmi_thread_t dmi_thread, int32_t dmi_id, dmi_thread_t *dmi_thread_ptr, int8_t *migrate_flag_ptr, status_t *status);
int32_t spi_isyield_thread(dmi_t *dmi, int8_t *yield_flag_ptr);
int32_t spi_yield_thread(dmi_t *dmi, int8_t *yield_flag_ptr);
void spi_wake_thread(dmi_t *dmi, dmi_thread_t dmi_thread, void *out_ptr, int64_t out_size, status_t *status);
int32_t spi_suspend_thread(dmi_t *dmi, void *in_ptr);
int32_t spi_self_thread(dmi_t *dmi, dmi_thread_t *dmi_thread_ptr);
void* spi_mmap_tls(dmi_t *dmi, void *start, size_t length, int prot, int flags, int fd, off_t offset);
void* spi_mremap_tls(dmi_t *dmi, void *old_address, size_t old_size, size_t new_size, int flags);
int32_t spi_mprotect_tls(dmi_t *dmi, void *addr, size_t len, int prot);
int32_t spi_munmap_tls(dmi_t *dmi, void *start, size_t length);
void spi_mmap_vms(dmi_t *dmi, int64_t *dmi_addrs, int64_t *page_sizes, int64_t *page_nums, int32_t vm_num, status_t *status);
void spi_munmap_vms(dmi_t *dmi, int64_t *dmi_addrs, int32_t vm_num, status_t *status);
int32_t spi_init_group(dmi_t *dmi, group_t *group, int64_t *addrs, int64_t *ptr_offsets, int64_t *sizes, int32_t group_num);
int32_t spi_destroy_group(dmi_t *dmi, group_t *group);
void spi_read_addr(dmi_t *dmi, int64_t dmi_addr, int64_t size, void *in_ptr, int8_t access_type, status_t *status);
void spi_gread_addr(dmi_t *dmi, group_t *group, void *in_ptr, int8_t access_type, status_t *status);
void spi_watch_addr(dmi_t *dmi, int64_t dmi_addr, int64_t size, void *in_ptr, void *out_ptr, status_t *status);
void spi_write_addr(dmi_t *dmi, int64_t dmi_addr, int64_t size, void *out_ptr, int8_t access_type, status_t *status);
void spi_gwrite_addr(dmi_t *dmi, group_t *group, void *out_ptr, int8_t access_type, status_t *status);
void spi_atomic_addr(dmi_t *dmi, int64_t dmi_addr, int64_t size, void *out_ptr, int64_t out_size, void *in_ptr, int64_t in_size, int8_t tag, int8_t access_type, status_t *status);
int32_t spi_save_addr(dmi_t *dmi, int64_t dmi_addr, int64_t size);
int32_t spi_unsave_addr(dmi_t *dmi, int64_t dmi_addr, int64_t size);
void spi_new_object(dmi_t *dmi, int32_t *object_ids, int64_t *slide_sizes, int32_t object_num, status_t *status);
void spi_delete_object(dmi_t *dmi, int32_t *object_ids, int32_t object_num, status_t *status);
void spi_read_object(dmi_t *dmi, int32_t object_id, int64_t object_offset, int64_t size, void *in_ptr, int8_t access_type, status_t *status);
void spi_write_object(dmi_t *dmi, int32_t object_id, int64_t object_offset, int64_t size, void *out_ptr, int8_t access_type, status_t *status);
void spi_gread_object(dmi_t *dmi, int32_t object_id, int64_t *object_offsets, int64_t *sizes, int64_t *ptr_offsets, int32_t group_num, void *in_ptr, int8_t access_type, status_t *status);
void spi_gwrite_object(dmi_t *dmi, int32_t object_id, int64_t *object_offsets, int64_t *sizes, int64_t *ptr_offsets, int32_t group_num, void *out_ptr, int8_t access_type, status_t *status);
void spi_atomic_object(dmi_t *dmi, int32_t object_id, void *out_ptr, int64_t out_size, void *in_ptr, int64_t in_size, int8_t tag, int8_t access_type, status_t *status);
void spi_watch_object(dmi_t *dmi, int32_t object_id, int64_t object_offset, int64_t size, void *in_ptr, void *out_ptr, status_t *status);
void spi_setsize_object(dmi_t *dmi, int32_t object_id, int64_t slide_size, int8_t access_type, status_t *status);
void spi_getsize_object(dmi_t *dmi, int32_t object_id, int64_t *slide_size_ptr, int8_t access_type, status_t *status);
int32_t spi_save_object(dmi_t *dmi, int32_t object_id);
int32_t spi_unsave_object(dmi_t *dmi, int32_t object_id);
void spi_start_status(status_t *status);
int32_t spi_check_status(status_t *status, int32_t *ret_ptr);
void spi_wait_status(status_t *status, int32_t *ret_ptr);

/* init/destroy/rank dmi */

void __attribute__((constructor)) constructor(void);
void __attribute__((destructor)) destructor(void);
void* dmi_monitor(void *arg);
void dmi_init(dmi_t *dmi, uint16_t listen_port, int64_t gmemory_size);
void dmi_destroy(dmi_t *dmi);
int32_t dmi_rank_dmi(dmi_t *dmi, int32_t *dmi_id_ptr);

/* create/join/detach/wake/suspend/self thread */

void dmi_create_thread(dmi_t *dmi, dmi_thread_t *dmi_thread_ptr, int32_t dmi_id, int64_t dmi_addr, int64_t stack_size, int8_t pheap_flag, int64_t value1, int64_t value2, int64_t value3, status_t *status);
void context_create_thread_pre(context_t *context);
void dmi_handle_req_create_thread(dmi_t *dmi, packet_t *recv_packet);
void dmi_call_thread(int32_t reg1, int32_t reg2);
void* context_switcher(void *arg);
int32_t dmi_isyield_thread(dmi_t *dmi, int8_t *yield_flag_ptr);
int32_t dmi_yield_thread(dmi_t *dmi, int8_t *yield_flag_ptr);
void context_yield_thread_pre(context_t *context);
void dmi_handle_req_yield_thread(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_yield_thread(dmi_t *dmi, packet_t *recv_packet);
void context_yield_thread_post(context_t *context);
void context_close_thread_pre(context_t *context);
void context_close_thread_post(context_t *context);
void dmi_handle_ack_create_thread(dmi_t *dmi, packet_t *recv_packet);
void context_create_thread_post(context_t *context);
void dmi_join_thread(dmi_t *dmi, dmi_thread_t dmi_thread, int64_t *dmi_addr_ptr, status_t *status);
void context_join_thread_pre(context_t *context);
void dmi_handle_req_join_thread(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_join_thread(dmi_t *dmi, packet_t *recv_packet);
void context_join_thread_post(context_t *context);
void dmi_detach_thread(dmi_t *dmi, dmi_thread_t dmi_thread, status_t *status);
void context_detach_thread_pre(context_t *context);
void dmi_handle_req_detach_thread(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_detach_thread(dmi_t *dmi, packet_t *recv_packet);
void context_detach_thread_post(context_t *context);
void dmi_migrate_thread(dmi_t *dmi, dmi_thread_t dmi_thread, int32_t dmi_id, dmi_thread_t *dmi_thread_ptr, int8_t *migrate_flag_ptr, status_t *status);
void context_migrate_thread_pre(context_t *context);
void dmi_handle_req_migrate_thread(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_migrate_thread(dmi_t *dmi, packet_t *recv_packet);
void context_migrate_thread_post(context_t *context);
void dmi_wake_thread(dmi_t *dmi, dmi_thread_t dmi_thread, void *out_ptr, int64_t out_size, status_t *status);
void context_wake_thread_pre(context_t *context);
void dmi_handle_req_wake_thread(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_wake_thread(dmi_t *dmi, packet_t *recv_packet);
void context_wake_thread_post(context_t *context);
int32_t dmi_suspend_thread(dmi_t *dmi, void *in_ptr);
int32_t dmi_self_thread(dmi_t *dmi, dmi_thread_t *dmi_thread_ptr);
void* dmi_mmap_tls(dmi_t *dmi, void *start, size_t length, int prot, int flags, int fd, off_t offset);
void* dmi_mremap_tls(dmi_t *dmi, void *old_address, size_t old_size, size_t new_size, int flags);
int32_t dmi_mprotect_tls(dmi_t *dmi, void *addr, size_t len, int prot);
int32_t dmi_munmap_tls(dmi_t *dmi, void *start, size_t length);

/* lock/unlock/trylock/delegate token */

void context_lock_token_pre(context_t *context);
void dmi_handle_req_lock_token(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_lock_token(dmi_t *dmi, packet_t *recv_packet);
void context_lock_token_post(context_t *context);
void context_unlock_token_pre(context_t *context);
void dmi_handle_req_unlock_token(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_unlock_token(dmi_t *dmi, packet_t *recv_packet);
void context_unlock_token_post(context_t *context);
void context_delegate_token_pre(context_t *context);
int32_t dmi_enter_token_heir(dmi_t *dmi, token_t *token);
void dmi_escape_token_heir(dmi_t *dmi, int32_t owner_id);
void dmi_handle_req_delegate_token(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_delegate_token(dmi_t *dmi, packet_t *recv_packet);
void context_delegate_token_post(context_t *context);

/* join/leave world */

void dmi_join_world(dmi_t *dmi, char *ip, uint16_t port, status_t *status);
void context_hello_world_pre(context_t *context);
void dmi_handle_req_hello_world(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_hello_world(dmi_t *dmi, packet_t *recv_packet);
void context_hello_world_post(context_t *context);
void context_alloc_peer_pre(context_t *context);
void dmi_handle_req_alloc_peer(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_alloc_peer(dmi_t *dmi, packet_t *recv_packet);
void context_alloc_peer_post(context_t *context);
void context_permit_peer_pre(context_t *context);
void dmi_handle_req_permit_peer(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_permit_peer(dmi_t *dmi, packet_t *recv_packet);
void context_permit_peer_post(context_t *context);
void dmi_leave_world(dmi_t *dmi, status_t *status);
void context_close_contexts_pre(context_t *context);
void context_close_contexts_post(context_t *context);
void context_forbid_peer_pre(context_t *context);
void dmi_handle_req_forbid_peer(dmi_t *dmi, packet_t *recv_packet);
void context_close_peer_pre(context_t *context);
void context_close_peer_post(context_t *context);
void dmi_handle_ack_forbid_peer(dmi_t *dmi, packet_t *recv_packet);
void context_forbid_peer_post(context_t *context);
void context_clear_pages_pre(context_t *context);
void context_clear_pages_post(context_t *context);
void context_free_peer_pre(context_t *context);
void dmi_handle_req_free_peer(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_free_peer(dmi_t *dmi, packet_t *recv_packet);
void context_free_peer_post(context_t *context);
void context_bye_world_pre(context_t *context);
void dmi_handle_req_bye_world(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_bye_world(dmi_t *dmi, packet_t *recv_packet);
void context_bye_world_post(context_t *context);

/* alloc/free vm */

void dmi_mmap_vms(dmi_t *dmi, int64_t *dmi_addrs, int64_t *page_sizes, int64_t *page_nums, int32_t vm_num, status_t *status);
void context_alloc_vms_pre(context_t *context);
void dmi_handle_req_alloc_vms(dmi_t *dmi, packet_t *recv_packet);
void dmi_alloc_pages(dmi_t *dmi, vm_t *vm, int32_t *dmi_ids, int32_t dmi_num, int64_t slide_size);
void dmi_handle_ack_alloc_vms(dmi_t *dmi, packet_t *recv_packet);
void context_alloc_vms_post(context_t *context);
void context_permit_vms_pre(context_t *context);
void dmi_handle_req_permit_vms(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_permit_vms(dmi_t *dmi, packet_t *recv_packet);
void context_permit_vms_post(context_t *context);
void dmi_munmap_vms(dmi_t *dmi, int64_t *dmi_addrs, int32_t vm_num, status_t *status);
void context_forbid_vms_pre(context_t *context);
void dmi_handle_req_forbid_vms(dmi_t *dmi, packet_t *recv_packet);
void context_close_vms_pre(context_t *context);
void context_close_vms_post(context_t *context);
void dmi_handle_ack_forbid_vms(dmi_t *dmi, packet_t *recv_packet);
void context_forbid_vms_post(context_t *context);
void context_free_vms_pre(context_t *context);
void dmi_handle_req_free_vms(dmi_t *dmi, packet_t *recv_packet);
void dmi_free_pages(dmi_t *dmi, vm_t *vm);
void dmi_handle_ack_free_vms(dmi_t *dmi, packet_t *recv_packet);
void context_free_vms_post(context_t *context);

/* read/write/gread/gwrite/atomic/decorate pages */

int32_t dmi_init_group(dmi_t *dmi, group_t *group, int64_t *addrs, int64_t *ptr_offsets, int64_t *sizes, int32_t group_num);
void sub_fastsort_vmids(int32_t n, int32_t *vm_ids);
void sub_fastsort_ranges(int32_t n, range_t *ranges);
int32_t dmi_destroy_group(dmi_t *dmi, group_t *group);
void dmi_read_addr(dmi_t *dmi, int64_t dmi_addr, int64_t size, void *in_ptr, int8_t access_type, status_t *status);
void dmi_gread_addr(dmi_t *dmi, group_t *group, void *in_ptr, int8_t access_type, status_t *status);
void dmi_write_addr(dmi_t *dmi, int64_t dmi_addr, int64_t size, void *out_ptr, int8_t access_type, status_t *status);
void dmi_gwrite_addr(dmi_t *dmi, group_t *group, void *out_ptr, int8_t access_type, status_t *status);
void dmi_watch_addr(dmi_t *dmi, int64_t dmi_addr, int64_t size, void *in_ptr, void *out_ptr, status_t *status);
void dmi_atomic_addr(dmi_t *dmi, int64_t dmi_addr, int64_t size, void *out_ptr, int64_t out_size, void *in_ptr, int64_t in_size, int8_t tag, int8_t access_type, status_t *status);
int32_t dmi_decorate_addr(dmi_t *dmi, int64_t dmi_addr, int64_t size, int8_t decor);
void dmi_new_object(dmi_t *dmi, int32_t *object_ids, int64_t *slide_sizes, int32_t object_num, status_t *status);
void dmi_delete_object(dmi_t *dmi, int32_t *object_ids, int32_t object_num, status_t *status);
void dmi_read_object(dmi_t *dmi, int32_t object_id, int64_t object_offset, int64_t size, void *in_ptr, int8_t access_type, status_t *status);
void dmi_write_object(dmi_t *dmi, int32_t object_id, int64_t object_offset, int64_t size, void *out_ptr, int8_t access_type, status_t *status);
void dmi_gread_object(dmi_t *dmi, int32_t object_id, int64_t *object_offsets, int64_t *sizes, int64_t *ptr_offsets, int32_t group_num, void *in_ptr, int8_t access_type, status_t *status);
void dmi_gwrite_object(dmi_t *dmi, int32_t object_id, int64_t *object_offsets, int64_t *sizes, int64_t *ptr_offsets, int32_t group_num, void *out_ptr, int8_t access_type, status_t *status);
void dmi_atomic_object(dmi_t *dmi, int32_t object_id, void *out_ptr, int64_t out_size, void *in_ptr, int64_t in_size, int8_t tag, int8_t access_type, status_t *status);
void dmi_watch_object(dmi_t *dmi, int32_t object_id, int64_t object_offset, int64_t size, void *in_ptr, void *out_ptr, status_t *status);
void dmi_getsize_object(dmi_t *dmi, int32_t object_id, int64_t *slide_size_ptr, int8_t access_type, status_t *status);
void dmi_setsize_object(dmi_t *dmi, int32_t object_id, int64_t slide_size, int8_t access_type, status_t *status);;
int32_t dmi_decorate_object(dmi_t *dmi, int32_t object_id, int8_t decor);
void context_access_page_pre(context_t *context);
void context_access_object_pre(context_t *context);
void context_access_pages_pre(context_t *context);
void context_gaccess_pages_pre(context_t *context);
void context_gaccess_object_pre(context_t *context);
void context_read_page_pre(context_t *context);
void sub_notify_watch(page_t *page);
void dmi_handle_req_read_page(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_req_relay_page(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_read_page(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_relay_page(dmi_t *dmi, packet_t *recv_packet);
void context_read_page_post(context_t *context);
void context_access_page_post(context_t *context);
void context_access_object_post(context_t *context);
void context_access_pages_post(context_t *context);
void context_gaccess_page_post(context_t *context);
void context_gaccess_object_post(context_t *context);
void context_gaccess_pages_post(context_t *context);
void context_exclude_page_pre(context_t *context);
void dmi_handle_req_steal_page(dmi_t *dmi, packet_t *recv_packet);
void context_migrate_page_pre(context_t *context);
void dmi_handle_req_migrate_page(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_migrate_page(dmi_t *dmi, packet_t *recv_packet);
void context_migrate_page_post(context_t *context);
void dmi_handle_ack_steal_page(dmi_t *dmi, packet_t *recv_packet);
void context_exclude_page_post(context_t *context);
void context_adjust_page_pre(context_t *context);
void dmi_handle_req_validate_page(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_validate_page(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_req_invalidate_page(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_invalidate_page(dmi_t *dmi, packet_t *recv_packet);
void context_adjust_page_post(context_t *context);
void dmi_handle_req_commit_page(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_commit_page(dmi_t *dmi, packet_t *recv_packet);
int32_t sub_check_order(dmi_t *dmi, order_t *order, packet_t *recv_packet);
void sub_increment_order(dmi_t *dmi, page_t *page);
account_t* sub_lookup_account(vector_t *account_vector, int32_t dmi_id);

/* sweep page */

void* dmi_sweeper(void *arg);
void dmi_order_vms(dmi_t *dmi, vector_t *vm_id_vector);
void dmi_kill_sweeper(dmi_t *dmi);
void context_sweep_page_pre(context_t *context);
void dmi_handle_req_sweep_page(dmi_t *dmi, packet_t *recv_packet);
int32_t dmi_enter_page_heir(dmi_t *dmi, page_t *page);
void dmi_escape_page_heir(dmi_t *dmi, int32_t owner_id);
void context_delegate_page_pre(context_t *context);
void dmi_handle_req_delegate_page(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_ack_delegate_page(dmi_t *dmi, packet_t *recv_packet);
void context_delegate_page_post(context_t *context);
void dmi_handle_ack_sweep_page(dmi_t *dmi, packet_t *recv_packet);
void context_sweep_page_post(context_t *context);

/* methods */

void* dmi_recver(void *arg);
void dmi_kill_recver(dmi_t *dmi);
void* dmi_handler(void *arg);
void dmi_kill_handler(dmi_t *dmi);
int64_t dmi_estimate_packet(dmi_t *dmi, packet_t *recv_packet);
void dmi_handle_packet(dmi_t *dmi, packet_t *recv_packet);
void dmi_send_packet(dmi_t *dmi, packet_t *send_packet);
void dmi_bcast_packet(dmi_t *dmi, packet_t *send_packet, int32_t *peer_num_ptr);
context_t* dmi_alloc_context(dmi_t *dmi, status_t *status, int8_t type);
void dmi_free_context(dmi_t *dmi, context_t *context, int8_t ret_flag);
context_t* dmi_get_context(dmi_t *dmi, int32_t context_id);

/* memory management */

void syscall_init(void);
void syscall_destroy(void);
void syscall_hijack(char *syscall_name, int32_t syscall_header_size, void *hook_syscall, void *true_syscall);
void* fixed_mmap(void *start, size_t length, int prot, int flags);
void* hook_syscall_mmap(void *start, size_t length, int prot, int flags, int fd, off_t offset);
void* hook_syscall_mremap(void *old_address, size_t old_size, size_t new_size, int flags);
int hook_syscall_munmap(void *start, size_t length);
int hook_syscall_mprotect(void *addr, size_t len, int prot);
int hook_syscall_brk(void *end_data_segment);
void print_int(int64_t d);
void print_intx(int64_t d);
void print_str(char *str);

/* private objects */

comm_t* comm_alloc(int16_t listen_port);
void comm_free(comm_t *comm);
void comm_disable(comm_t *comm);
int32_t comm_recv(comm_t *comm, packet_t *recv_packet);
void comm_send(comm_t *comm, int32_t src_dmi_id, int32_t dst_dmi_id, packet_t *send_packet);
void comm_connect(comm_t *comm, int32_t src_dmi_id, int32_t dst_dmi_id, char *ip, int16_t port);
void comm_close(comm_t *comm, int32_t dmi_id);
void comm_add(comm_t *comm, int32_t dmi_id, int sock);
int comm_delete(comm_t *comm, int32_t dmi_id);
void comm_self(comm_t *comm, host_t *host_ptr);
packet_t* packet_alloc(void);
void packet_free(packet_t *packet);
void packet_copy(packet_t *dst_packet, packet_t *src_packet);
void packet_send(packet_t *send_packet, int sock, int8_t type);
int32_t packet_recv(packet_t *recv_packet, int sock, int8_t type);
void packet_print(packet_t *packet, int8_t type);
gmemory_t* gmemory_alloc(int64_t max_size);
void gmemory_free(gmemory_t *gmemory);
void* gmemory_change(gmemory_t *gmemory, void *buf, int64_t *buf_size, int64_t size);
int32_t gmemory_monitor(gmemory_t *gmemory, int64_t *total_size_ptr, int64_t *max_size_ptr);
void gmemory_disable(gmemory_t *gmemory);
gate_t* gate_alloc(void);
void gate_free(gate_t *gate);
void gate_permit(gate_t *gate);
void gate_forbid(gate_t *gate, context_t *context, void (*func_ptr)(context_t *context));
int32_t gate_enter(gate_t *gate);
void gate_escape(gate_t *gate);
order_t* order_alloc(void);
void order_free(order_t *order);
int32_t order_check(order_t *order, packet_t *recv_packet);
packet_t* order_increment(order_t *order);
load_t* load_alloc(void);
void load_free(load_t *load);
void load_increase(load_t *load, int64_t size);
void load_decrease(load_t *load, int64_t size);
int64_t load_size(load_t *load);
async_t* async_alloc(void);
void async_free(async_t *async);
void async_finish(async_t *async, int32_t ret);
int32_t async_check(async_t *async, int32_t *ret_ptr);
void async_wait(async_t *async, int32_t *ret_ptr);
status_t* status_alloc(void);
void status_free(status_t *status);
void status_start(status_t *status);
void status_finish(status_t *status, int32_t ret);
int32_t status_check(status_t *status, int32_t *ret_ptr);
void status_wait(status_t *status, int32_t *ret_ptr);
window_t* window_alloc(int32_t window_size);
void window_free(window_t *window);
status_t* window_expand(window_t *window);
void window_flush(window_t *window);

/* public objects */

dmi_t* dmi_alloc(void);
void dmi_free(dmi_t *dmi);
token_t* token_alloc(void);
void token_free(token_t *token);
peer_t* peer_alloc(void);
void peer_free(peer_t *peer);
thread_t* thread_alloc(void);
void thread_free(thread_t *thread);
vm_t* vm_alloc(void);
void vm_free(vm_t *vm);
page_t* page_alloc(void);
void page_free(page_t *page);
account_t* account_alloc(void);
void account_free(account_t *account);
pheap_t* pheap_alloc(void);
void pheap_free(pheap_t *pheap);
mblock_t* mblock_alloc(void);
void mblock_free(mblock_t *mblock);
mmapped_t* mmapped_alloc(void);
void mmapped_free(mmapped_t *mmapped);
context_t* context_alloc(void);
void context_free(context_t *context);


#endif
