#include "dmi_api.h"

int _ticket_channel[2];
int _scheduler_channel[2];
wdmi_t *_wdmi;
dmi_mutex_t DMI_MUTEX_INITIALIZER = {DMI_NULL, DMI_NULL, DMI_NULL, DMI_NULL};
dmi_cond_t DMI_COND_INITIALIZER = {DMI_NULL, DMI_NULL, STATE_NONE};
__thread wdmi_thread_t *_tls_wdmi_thread;
__thread int8_t _tls_thread_malloc_flag;
__thread int _tls_profile_flag;
__thread int8_t _tls_profile_rec_flag;

__thread int64_t _dmi_memory = 0;

/* main */

int main(int argc, char **argv)
{
  ticket_t *ticket;
  status_t status;
  dmi_node_t dmi_node;
  dmi_thread_t main_thread;
  wdmi_thread_t *wdmi_thread;
  stackarea_t stackarea;
  mheader_t *mheader;
  struct sigaction action;
  int32_t ret, dmi_id, node_num;
  
  if(sizeof(void*) != 8)
    {
      fprintf(stderr, "DMI works only on 64bit Linux architectures!\n");
      exit(1);
    }
  
  ret = pipe(_ticket_channel);
  if(ret < 0) error();
  ret = pipe(_scheduler_channel);
  if(ret < 0) error();
  
  memset(&action, 0, sizeof(struct sigaction));
  action.sa_sigaction = signal_handler;
  action.sa_flags = SA_SIGINFO;
  ret = sigaction(SIGINT, &action, NULL);
  if(ret < 0) error();
  
  _tls_profile_flag = FALSE;
  
  _wdmi = wdmi_alloc();
  
  _wdmi->profile_flag = FALSE;
  
  _wdmi->config = config_alloc(argc, argv);
  
  _wdmi->dmi = spi_alloc_dmi(_wdmi->config->listen_port, _wdmi->config->mem_size);
  
  //fprintf(stderr, "=== initialized ===\n");
  
  spi_start_status(&status);
  spi_join_world(_wdmi->dmi, _wdmi->config->ip, _wdmi->config->port, &status);
  spi_wait_status(&status, &ret);
  catch(ret);
  
  ret = DMI_rank(&dmi_id);
  catch(ret);
  
  if(_wdmi->config->ip == NULL)
    {
      ret = DMI_mmap(&_wdmi->main_addr, sizeof(int64_t) * 8, 1, NULL);
      catch(ret);
      ret = DMI_mmap(&_wdmi->dmi_member_addr, sizeof(dmi_member_t), 1, NULL);
      catch(ret);
      ret = DMI_mmap(&_wdmi->dmi_mthread_addr, sizeof(dmi_mthread_t), 1, NULL);
      catch(ret);
      ret = DMI_mmap(&_wdmi->dmi_memalloc_addr, sizeof(dmi_memalloc_t), 1, NULL);
      catch(ret);
      ret = DMI_mmap(&_wdmi->dmi_forkkill_addr, sizeof(dmi_forkkill_t), 1, NULL);
      catch(ret);
      ret = DMI_mmap(&_wdmi->dmi_profile_addr, sizeof(dmi_profile_t), 1, NULL);
      catch(ret);
      
      ret = dmi_member_init(_wdmi->dmi_member_addr);
      catch(ret);
      ret = dmi_mthread_init(_wdmi->dmi_mthread_addr);
      catch(ret);
      ret = dmi_memalloc_init(_wdmi->dmi_memalloc_addr);
      catch(ret);
      ret = dmi_forkkill_init(_wdmi->dmi_forkkill_addr);
      catch(ret);
      ret = dmi_profile_init(_wdmi->dmi_profile_addr, _wdmi->config->profile_type, _wdmi->config->profile_name);
      catch(ret);
      
      ret = DMI_write(_wdmi->main_addr, sizeof(int64_t), &_wdmi->dmi_member_addr, DMI_EXCLUSIVE, NULL);
      catch(ret);
      ret = DMI_write(_wdmi->main_addr + sizeof(int64_t), sizeof(int64_t), &_wdmi->dmi_profile_addr, DMI_EXCLUSIVE, NULL);
      catch(ret);
      ret = DMI_write(_wdmi->main_addr + 2 * sizeof(int64_t), sizeof(int64_t), &_wdmi->dmi_mthread_addr, DMI_EXCLUSIVE, NULL);
      catch(ret);
      ret = DMI_write(_wdmi->main_addr + 3 * sizeof(int64_t), sizeof(int64_t), &_wdmi->dmi_memalloc_addr, DMI_EXCLUSIVE, NULL);
      catch(ret);
      ret = DMI_write(_wdmi->main_addr + 4 * sizeof(int64_t), sizeof(int64_t), &_wdmi->dmi_forkkill_addr, DMI_EXCLUSIVE, NULL);
      catch(ret);
      
      ret = DMI_write(DMI_BOOT_ADDR, sizeof(int64_t), &_wdmi->main_addr, DMI_EXCLUSIVE, NULL);
      catch(ret);
    }
  else
    {
      ret = DMI_read(DMI_BOOT_ADDR, sizeof(int64_t), &_wdmi->main_addr, DMI_GET, NULL);
      catch(ret);
      
      ret = DMI_read(_wdmi->main_addr, sizeof(int64_t), &_wdmi->dmi_member_addr, DMI_GET, NULL);
      catch(ret);
      ret = DMI_read(_wdmi->main_addr + sizeof(int64_t), sizeof(int64_t), &_wdmi->dmi_profile_addr, DMI_GET, NULL);
      catch(ret);
      ret = DMI_read(_wdmi->main_addr + 2 * sizeof(int64_t), sizeof(int64_t), &_wdmi->dmi_mthread_addr, DMI_GET, NULL);
      catch(ret);
      ret = DMI_read(_wdmi->main_addr + 3 * sizeof(int64_t), sizeof(int64_t), &_wdmi->dmi_memalloc_addr, DMI_GET, NULL);
      catch(ret);
      ret = DMI_read(_wdmi->main_addr + 4 * sizeof(int64_t), sizeof(int64_t), &_wdmi->dmi_forkkill_addr, DMI_GET, NULL);
      catch(ret);
    }
  
  dmi_member_fad(_wdmi->dmi_member_addr, 1, NULL);
  catch(ret);
  
  wdmi_thread = wdmi_thread_alloc();
  ret = dmi_mthread_get(_wdmi->dmi_mthread_addr, &wdmi_thread->mthread_id, &wdmi_thread->stackarea_addr);
  catch(ret);
  
  mheader = wdmi_thread->dummy_mheader;
  mheader->next_mheader = mheader;
  mheader->size = 0;
  wdmi_thread->head_mheader = mheader;
  
  ret = DMI_self(&wdmi_thread->dmi_thread);
  catch(ret);
  
  _tls_wdmi_thread = wdmi_thread;
  
  ret = profile_init(_wdmi->profile, _wdmi->dmi_profile_addr);
  catch(ret);
  
  _wdmi->profile_flag = TRUE;
  
  stackarea.dmi_thread = wdmi_thread->dmi_thread;
  stackarea.next_addr = DMI_NULL;
  ret = DMI_write(wdmi_thread->stackarea_addr, sizeof(stackarea_t), &stackarea, DMI_EXCLUSIVE, NULL);
  catch(ret);
  
  ticket = ticket_alloc(_wdmi->dmi_forkkill_addr);
  
  //fprintf(stderr, "=== joined ===\n");
  
  if(_wdmi->config->ip == NULL)
    {
      ret = DMI_create2(&main_thread, dmi_id, DMI_NULL, 0, FALSE, THREAD_MAIN, (int64_t)_wdmi->config->argc, (int64_t)_wdmi->config->argv, NULL);
      catch(ret);
    }
  
  ret = dmi_member_join(_wdmi->dmi_member_addr, dmi_id, _wdmi->config->core_num, _wdmi->config->mem_size, &dmi_node, _wdmi->config->detach_flag);
  catch(ret);
  
  if(_wdmi->config->detach_flag == TRUE)
    {
      ret = dmi_forkkill_consumepost(_wdmi->dmi_forkkill_addr, dmi_node);
      catch(ret);
    }
  
  //fprintf(stderr, "=== opened ===\n");
  
  _wdmi->config->detach_flag = FALSE;
  
  ticket_monitor(ticket);
  
  //fprintf(stderr, "=== closed ===\n");
  
  if(_wdmi->config->detach_flag == TRUE)
    {
      ret = dmi_forkkill_consumepost(_wdmi->dmi_forkkill_addr, dmi_node);
      catch(ret);
    }
  
  ret = dmi_member_leave(_wdmi->dmi_member_addr, dmi_id, _wdmi->config->detach_flag);
  catch(ret);
  
  if(_wdmi->config->ip == NULL)
    {
      ret = DMI_join(main_thread, NULL, NULL);
      catch(ret);
    }
  
  ticket_joinchild(ticket);
  
  ticket_free(ticket);
  
  _wdmi->profile_flag = FALSE;
  
  ret = profile_destroy(_wdmi->profile);
  catch(ret);
  
  ret = dmi_mthread_put(_wdmi->dmi_mthread_addr, wdmi_thread->mthread_id);
  catch(ret);
  wdmi_thread_free(wdmi_thread);
  
  _tls_wdmi_thread = NULL;
  
  dmi_member_fad(_wdmi->dmi_member_addr, -1, &node_num);
  catch(ret);
  
  if(node_num == 1)
    {
      ret = dmi_profile_destroy(_wdmi->dmi_profile_addr);
      catch(ret);
      ret = dmi_forkkill_destroy(_wdmi->dmi_forkkill_addr);
      catch(ret);
      ret = dmi_memalloc_destroy(_wdmi->dmi_memalloc_addr);
      catch(ret);
      ret = dmi_mthread_destroy(_wdmi->dmi_mthread_addr);
      catch(ret);
      ret = dmi_member_destroy(_wdmi->dmi_member_addr);
      catch(ret);
      
      ret = DMI_munmap(_wdmi->dmi_profile_addr, NULL);
      catch(ret);
      ret = DMI_munmap(_wdmi->dmi_forkkill_addr, NULL);
      catch(ret);
      ret = DMI_munmap(_wdmi->dmi_memalloc_addr, NULL);
      catch(ret);
      ret = DMI_munmap(_wdmi->dmi_mthread_addr, NULL);
      catch(ret);
      ret = DMI_munmap(_wdmi->dmi_member_addr, NULL);
      catch(ret);
      ret = DMI_munmap(_wdmi->main_addr, NULL);
      catch(ret);
    }
  
  spi_start_status(&status);
  spi_leave_world(_wdmi->dmi, &status);
  spi_wait_status(&status, &ret);
  
  //fprintf(stderr, "=== left ===\n");
  
  spi_free_dmi(_wdmi->dmi);
  
  //fprintf(stderr, "=== finalized ===\n");
  
  config_free(_wdmi->config);
  
  wdmi_free(_wdmi);
  
  close(_scheduler_channel[PIPE_IN]);
  close(_scheduler_channel[PIPE_OUT]);
  close(_ticket_channel[PIPE_IN]);
  close(_ticket_channel[PIPE_OUT]);
  return 0;
}

int64_t dmi_thread(int64_t dmi_addr, int64_t value1, int64_t value2, int64_t value3)
{
  dmi_node_t *dmi_nodes;
  wdmi_thread_t *wdmi_thread;
  int64_t ret_dmi_addr;
  stackarea_t stackarea;
  mheader_t *mheader;
  int32_t ret, dmi_node_id, dmi_node_num, node_num;
  
  wdmi_thread = wdmi_thread_alloc();
  ret = dmi_mthread_get(_wdmi->dmi_mthread_addr, &wdmi_thread->mthread_id, &wdmi_thread->stackarea_addr);
  catch(ret);
  
  mheader = wdmi_thread->dummy_mheader;
  mheader->next_mheader = mheader;
  mheader->size = 0;
  wdmi_thread->head_mheader = mheader;
  
  ret = DMI_self(&wdmi_thread->dmi_thread);
  catch(ret);
  
  _tls_wdmi_thread = wdmi_thread;
  
  stackarea.dmi_thread = wdmi_thread->dmi_thread;
  stackarea.next_addr = DMI_NULL;
  ret = DMI_write(wdmi_thread->stackarea_addr, sizeof(stackarea_t), &stackarea, DMI_EXCLUSIVE, NULL);
  catch(ret);
  
  _tls_profile_flag = TRUE;
  
  ret_dmi_addr = DMI_NULL;
  if(value1 == THREAD_MAIN)
    {
      if(DMI_main)
        {
          DMI_main((int32_t)value2, (char**)value3);
          ret_dmi_addr = DMI_NULL;
        }
      else
        {
          fprintf(stderr, "error : DMI_main is not defined!\n");
          error();
        }
      
            
      dmi_member_rescalefad(_wdmi->dmi_member_addr, 0, &node_num);
      catch(ret);
      
      dmi_nodes = (dmi_node_t*)my_malloc(sizeof(dmi_node_t) * CONST_NODE_MAX);
      while(node_num > 0)
        {
          ret = DMI_nodes(dmi_nodes, &dmi_node_num, CONST_NODE_MAX);
          catch(ret);
          
          for(dmi_node_id = 0; dmi_node_id < dmi_node_num; dmi_node_id++)
            {
              if(dmi_nodes[dmi_node_id].state == DMI_OPEN)
                {
                  node_num++;
                  ret = DMI_welcome(dmi_nodes[dmi_node_id].dmi_id);
                  catch(ret);
                }
              else if(dmi_nodes[dmi_node_id].state == DMI_CLOSE)
                {
                  node_num--;
                  ret = DMI_goodbye(dmi_nodes[dmi_node_id].dmi_id);
                  catch(ret);
                }
            }
          halt(CONST_RESCALE_HALT_INTERVAL);
        }
      my_free(dmi_nodes);
    }
  else if(value1 == THREAD_PROXY)
    {
      ticket_proxyer2((ticket_t*)dmi_addr);
    }
  else if(value1 == THREAD_SCHEDULER_A)
    {
      dmi_scheduler_monitor(dmi_addr);
      ret_dmi_addr = DMI_NULL;
    }
  else if(value1 == THREAD_NORMAL)
    {
      if(DMI_thread)
        {
          ret_dmi_addr = DMI_thread(dmi_addr);
        }
      else
        {
          fprintf(stderr, "error : DMI_thread is not defined!\n");
          error();
        }
    }
  else if(value1 == THREAD_SCALEUNIT)
    {
      ret = dmi_rescale_main(dmi_addr);
      catch(ret);
    }
  else if(value1 == THREAD_SCHEDULED)
    {
      ret = dmi_scheduler_main(dmi_addr, (int32_t)value2, value3);
      catch(ret);
    }
  else
    {
      error();
    }
  
  wdmi_thread = _tls_wdmi_thread;
  
  _tls_profile_flag = FALSE;
  
  _tls_wdmi_thread = NULL;
  
  ret = dmi_mthread_put(_wdmi->dmi_mthread_addr, wdmi_thread->mthread_id);
  catch(ret);
  wdmi_thread_free(wdmi_thread);
  return ret_dmi_addr;
}

void dmi_function(void *page_ptr, int64_t copy_size, void *out_ptr, int64_t out_size, void *in_ptr, int64_t in_size, int8_t tag)
{
  int64_t uoffset, usize, size, object_offset;
  void *cmp_ptr, *swap_ptr, *out2_ptr;
  
  switch(tag)
    {
    case SYS_CAS_TAG:
      uoffset = 0;
      usize = copy_size;
      cmp_ptr = (int8_t*)out_ptr + uoffset;
      uoffset += usize;
      usize = copy_size;
      swap_ptr = (int8_t*)out_ptr + uoffset;
      uoffset += usize;
      
      if(memcmp(page_ptr, cmp_ptr, copy_size) == 0)
        {
          memcpy(page_ptr, swap_ptr, copy_size);
          if(in_size > 0)
            {
              ((int8_t*)in_ptr)[0] = TRUE;
            }
        }
      else
        {
          if(in_size > 0)
            {
              ((int8_t*)in_ptr)[0] = FALSE;
            }
        }
      break;
    case SYS_FAD_TAG:
      if(in_ptr != NULL)
        {
          *(int64_t*)in_ptr = *(int64_t*)page_ptr;
        }
      *(int64_t*)page_ptr += *(int64_t*)out_ptr;
      break;
    case SYS_FAS_TAG:
      if(in_size > 0)
        {
          memcpy(in_ptr, page_ptr, copy_size);
        }
      memcpy(page_ptr, out_ptr, copy_size);
      break;
    case SYS_CAS_OBJECT_TAG:
      uoffset = 0;
      usize = sizeof(int64_t);
      object_offset = *(int64_t*)((int8_t*)out_ptr + uoffset);
      uoffset += usize;
      usize = sizeof(int64_t);
      size = *(int64_t*)((int8_t*)out_ptr + uoffset);
      uoffset += usize;
      usize = size;
      cmp_ptr = (int8_t*)out_ptr + uoffset;
      uoffset += usize;
      usize = size;
      swap_ptr = (int8_t*)out_ptr + uoffset;
      uoffset += usize;
      
      if(memcmp(page_ptr, cmp_ptr, size) == 0)
        {
          memcpy(page_ptr, swap_ptr, size);
          if(in_size > 0)
            {
              ((int8_t*)in_ptr)[0] = TRUE;
            }
        }
      else
        {
          if(in_size > 0)
            {
              ((int8_t*)in_ptr)[0] = FALSE;
            }
        }
      break;
    case SYS_FAS_OBJECT_TAG:
      uoffset = 0;
      usize = sizeof(int64_t);
      object_offset = *(int64_t*)((int8_t*)out_ptr + uoffset);
      uoffset += usize;
      usize = sizeof(int64_t);
      size = *(int64_t*)((int8_t*)out_ptr + uoffset);
      uoffset += usize;
      usize = size;
      out2_ptr = (int8_t*)out_ptr + uoffset;
      uoffset += usize;
      
      if(in_size > 0)
        {
          memcpy(in_ptr, page_ptr, copy_size);
        }
      memcpy(page_ptr, out2_ptr, copy_size);
      break;
    case SYS_INT64_BARRIER_TAG:
      if(((int64_t*)page_ptr)[1] == 0)
        {
          switch((int8_t)((int64_t*)out_ptr)[2])
            {
            case DMI_OP_MAX:
              ((int64_t*)page_ptr)[0] = -INT64_MAX;
              break;
            case DMI_OP_MIN:
              ((int64_t*)page_ptr)[0] = INT64_MAX;
              break;
            case DMI_OP_SUM:
              ((int64_t*)page_ptr)[0] = 0;
              break;
            case DMI_OP_PROD:
              ((int64_t*)page_ptr)[0] = 1;
              break;
            default:
              error();
            }
        }
      
      switch((int8_t)((int64_t*)out_ptr)[2])
        {
        case DMI_OP_MAX:
          if(((int64_t*)page_ptr)[0] < ((int64_t*)out_ptr)[0])
            {
              ((int64_t*)page_ptr)[0] = ((int64_t*)out_ptr)[0];
            }
          break;
        case DMI_OP_MIN:
          if(((int64_t*)page_ptr)[0] > ((int64_t*)out_ptr)[0])
            {
              ((int64_t*)page_ptr)[0] = ((int64_t*)out_ptr)[0];
            }
          break;
        case DMI_OP_SUM:
          ((int64_t*)page_ptr)[0] += ((int64_t*)out_ptr)[0];
          break;
        case DMI_OP_PROD:
          ((int64_t*)page_ptr)[0] *= ((int64_t*)out_ptr)[0];
          break;
        default:
          error();
        }
      
      ((int64_t*)page_ptr)[1] += 1;
      if(((int64_t*)page_ptr)[1] == ((int64_t*)out_ptr)[1])
        {
          ((int64_t*)in_ptr)[0] = ((int64_t*)page_ptr)[0];
          ((int64_t*)in_ptr)[1] = ((int64_t*)page_ptr)[2];
          ((int64_t*)page_ptr)[1] = 0;
          ((int64_t*)page_ptr)[2] = ((int)((int64_t*)page_ptr)[2] + 1) % 3;
        }
      else
        {
          ((int64_t*)in_ptr)[0] = SYS_INT64_DUMMY;
          ((int64_t*)in_ptr)[1] = ((int64_t*)page_ptr)[2];
        }
      break;
    case SYS_DOUBLE_BARRIER_TAG:
      if(((double*)page_ptr)[1] == 0)
        {
          switch((int8_t)((double*)out_ptr)[2])
            {
            case DMI_OP_MAX:
              ((double*)page_ptr)[0] = SYS_DOUBLE_DUMMY;
              break;
            case DMI_OP_MIN:
              ((double*)page_ptr)[0] = SYS_DOUBLE_DUMMY;
              break;
            case DMI_OP_SUM:
              ((double*)page_ptr)[0] = 0;
              break;
            case DMI_OP_PROD:
              ((double*)page_ptr)[0] = 1;
              break;
            default:
              error();
            }
        }
      
      switch((int8_t)(int64_t)((double*)out_ptr)[2])
        {
        case DMI_OP_MAX:
          if(((double*)page_ptr)[0] == SYS_DOUBLE_DUMMY ||
             ((double*)page_ptr)[0] < ((double*)out_ptr)[0])
            {
              ((double*)page_ptr)[0] = ((double*)out_ptr)[0];
            }
          break;
        case DMI_OP_MIN:
          if(((double*)page_ptr)[0] == SYS_DOUBLE_DUMMY ||
             ((double*)page_ptr)[0] > ((double*)out_ptr)[0])
            {
              ((double*)page_ptr)[0] = ((double*)out_ptr)[0];
            }
          break;
        case DMI_OP_SUM:
          ((double*)page_ptr)[0] += ((double*)out_ptr)[0];
          break;
        case DMI_OP_PROD:
          ((double*)page_ptr)[0] *= ((double*)out_ptr)[0];
          break;
        default:
          error();
        }
      
      ((double*)page_ptr)[1] += 1;
      if(fabs(((double*)page_ptr)[1] - ((double*)out_ptr)[1]) < EPS12)
        {
          ((double*)in_ptr)[0] = ((double*)page_ptr)[0];
          ((double*)in_ptr)[1] = ((double*)page_ptr)[2];
          ((double*)page_ptr)[1] = 0;
          ((double*)page_ptr)[2] = ((int)((double*)page_ptr)[2] + 1) % 3;
        }
      else
        {
          ((double*)in_ptr)[0] = SYS_DOUBLE_DUMMY;
          ((double*)in_ptr)[1] = ((double*)page_ptr)[2];
        }
      break;
    default:
      if(DMI_function)
        {
          DMI_function(page_ptr, copy_size, out_ptr, out_size, in_ptr, in_size, tag);
        }
      else
        {
          fprintf(stderr, "error : DMI_function is not defined!\n");
          error();
        }
    }
  return;
}

/* wrapper api */

int32_t DMI_rank(int32_t *dmi_id_ptr)
{
  int32_t ret;
  
  ret = spi_rank_dmi(_wdmi->dmi, dmi_id_ptr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return ret;
}

int32_t DMI_create(dmi_thread_t *dmi_thread_ptr, int32_t dmi_id, int64_t dmi_addr, int64_t stack_size, status_t *status)
{
  int32_t ret;
  
  ret = DMI_create2(dmi_thread_ptr, dmi_id, dmi_addr, stack_size, FALSE, THREAD_NORMAL, 0, 0, status);
  return ret;
}

int32_t DMI_create2(dmi_thread_t *dmi_thread_ptr, int32_t dmi_id, int64_t dmi_addr, int64_t stack_size, int8_t pheap_flag, int64_t value1, int64_t value2, int64_t value3, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  spi_create_thread(_wdmi->dmi, dmi_thread_ptr, dmi_id, dmi_addr, stack_size, pheap_flag, value1, value2, value3, status);
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_join(dmi_thread_t dmi_thread, int64_t *dmi_addr_ptr, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  spi_join_thread(_wdmi->dmi, dmi_thread, dmi_addr_ptr, status);
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_detach(dmi_thread_t dmi_thread, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  spi_detach_thread(_wdmi->dmi, dmi_thread, status);
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_migrate(dmi_thread_t dmi_thread, int32_t dmi_id, dmi_thread_t *dmi_thread_ptr, int8_t *migrate_flag_ptr, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  spi_migrate_thread(_wdmi->dmi, dmi_thread, dmi_id, dmi_thread_ptr, migrate_flag_ptr, status);
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_yield(void)
{
  wdmi_thread_t *wdmi_thread;
  stackarea_t stackarea;
  mheader_t *mheader;
  int8_t yield_flag;
  int32_t ret;
  
  ret = spi_isyield_thread(_wdmi->dmi, &yield_flag);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  if(yield_flag == TRUE)
    {
      wdmi_thread = _tls_wdmi_thread;
      
      _tls_profile_flag = FALSE;
      
      _tls_wdmi_thread = NULL;
      
      ret = dmi_mthread_put(_wdmi->dmi_mthread_addr, wdmi_thread->mthread_id);
      catch(ret);
      wdmi_thread_free(wdmi_thread);
      
      ret = spi_yield_thread(_wdmi->dmi, &yield_flag);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      if(yield_flag == FALSE) error();
      
      wdmi_thread = wdmi_thread_alloc();
      ret = dmi_mthread_get(_wdmi->dmi_mthread_addr, &wdmi_thread->mthread_id, &wdmi_thread->stackarea_addr);
      catch(ret);
      
      mheader = wdmi_thread->dummy_mheader;
      mheader->next_mheader = mheader;
      mheader->size = 0;
      wdmi_thread->head_mheader = mheader;
      
      ret = DMI_self(&wdmi_thread->dmi_thread);
      catch(ret);
      
      _tls_wdmi_thread = wdmi_thread;
      
      stackarea.dmi_thread = wdmi_thread->dmi_thread;
      stackarea.next_addr = DMI_NULL;
      ret = DMI_write(wdmi_thread->stackarea_addr, sizeof(stackarea_t), &stackarea, DMI_EXCLUSIVE, NULL);
      catch(ret);
      
      _tls_profile_flag = TRUE;
    }
  return TRUE;
}

int32_t DMI_wake(dmi_thread_t dmi_thread, void *out_ptr, int64_t out_size, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  spi_wake_thread(_wdmi->dmi, dmi_thread, out_ptr, out_size, status);
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_suspend(void *in_ptr)
{
  int32_t ret;
  
  ret = spi_suspend_thread(_wdmi->dmi, in_ptr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_self(dmi_thread_t *dmi_thread_ptr)
{
  int32_t ret;
  
  ret = spi_self_thread(_wdmi->dmi, dmi_thread_ptr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_fork(char *option, int32_t dmi_id, dmi_node_t *dmi_node_ptr)
{
  int32_t ret;
  
  ret = dmi_forkkill_produce(_wdmi->dmi_forkkill_addr, COMMAND_FORK, dmi_id, dmi_node_ptr, option);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_kill(int32_t dmi_id)
{
  int32_t ret;
  
  ret = dmi_forkkill_produce(_wdmi->dmi_forkkill_addr, COMMAND_KILL, dmi_id, NULL, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_mmap(int64_t *dmi_addr_ptr, int64_t page_size, int64_t page_num, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(_wdmi->profile_flag == TRUE
     && profile_get_type(_wdmi->profile) == PROFILE_OUT
     && _tls_profile_flag == TRUE)
    {
      if(page_size == DMI_AUTOMATIC)
        {
          page_size = page_num < CONST_DEFAULT_PAGE_SIZE ? page_num : CONST_DEFAULT_PAGE_SIZE;
          if(page_num % page_size == 0)
            {
              page_num = page_num / page_size;
            }
          else
            {
              page_num = page_num / page_size + 1;
            }
        }
      
      if(status == NULL)
        {
          status = &tmp_status;
        }
      spi_start_status(status);
      
      spi_mmap_vms(_wdmi->dmi, dmi_addr_ptr, &page_size, &page_num, 1, status);
      
      if(status == &tmp_status)
        {
          spi_wait_status(status, &ret);
          throw_or_catch(_wdmi->config->catch_flag, ret);
        }
      
      ret = profile_record_mmap(_wdmi->profile, *dmi_addr_ptr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  else
    {
      if(page_size == DMI_AUTOMATIC)
        {
          if(_wdmi->profile_flag == TRUE
             && profile_get_type(_wdmi->profile) == PROFILE_IN)
            {
              ret = profile_decide_page_size(_wdmi->profile, &page_size);
              throw_or_catch(_wdmi->config->catch_flag, ret);
              if(page_size == 0)
                {
                  page_size = page_num;
                }
#if DEBUG_BASIC
              fprintf(stderr, "optimized page size : %ld\n", page_size);
#endif
            }
          else
            {
              page_size = page_num < CONST_DEFAULT_PAGE_SIZE ? page_num : CONST_DEFAULT_PAGE_SIZE;
            }
          
          if(page_num % page_size == 0)
            {
              page_num = page_num / page_size;
            }
          else
            {
              page_num = page_num / page_size + 1;
            }
        }
      
      if(status == NULL)
        {
          status = &tmp_status;
        }
      spi_start_status(status);
      
      spi_mmap_vms(_wdmi->dmi, dmi_addr_ptr, &page_size, &page_num, 1, status);
      
      if(status == &tmp_status)
        {
          spi_wait_status(status, &ret);
          throw_or_catch(_wdmi->config->catch_flag, ret);
        }
    }
  return TRUE;
}

int32_t DMI_munmap(int64_t dmi_addr, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(_wdmi->profile_flag == TRUE
     && profile_get_type(_wdmi->profile) == PROFILE_OUT
     && _tls_profile_flag == TRUE)
    {
      ret = profile_record_munmap(_wdmi->profile, dmi_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  spi_munmap_vms(_wdmi->dmi, &dmi_addr, 1, status);
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_group_init(group_t *group, int64_t *addrs, int64_t *ptr_offsets, int64_t *sizes, int32_t group_num)
{
  int32_t ret;
  
  ret = spi_init_group(_wdmi->dmi, group, addrs, ptr_offsets, sizes, group_num);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_group_destroy(group_t *group)
{
  int32_t ret;
  
  ret = spi_destroy_group(_wdmi->dmi, group);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_read(int64_t dmi_addr, int64_t size, void *in_ptr, int8_t access_type, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(_wdmi->profile_flag == TRUE
     && profile_get_type(_wdmi->profile) == PROFILE_OUT
     && _tls_profile_flag == TRUE
     && (access_type == DMI_INVALIDATE || access_type == DMI_UPDATE))
    {
      ret = profile_record_access(_wdmi->profile, dmi_addr, size);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
#if DEBUG_ACCESS
  int ran = mrand_int(0, 2);
  switch(ran)
    {
    case 0:
      spi_read_addr(_wdmi->dmi, dmi_addr, size, in_ptr, DMI_GET, status);
      break;
    case 1:
      spi_read_addr(_wdmi->dmi, dmi_addr, size, in_ptr, DMI_INVALIDATE, status);
      break;
    case 2:
      spi_read_addr(_wdmi->dmi, dmi_addr, size, in_ptr, DMI_GET, status);
      break;
    default:
      error();
    }
#else
  spi_read_addr(_wdmi->dmi, dmi_addr, size, in_ptr, access_type, status);
#endif
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_group_read(group_t *group, void *in_ptr, int8_t access_type, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  spi_gread_addr(_wdmi->dmi, group, in_ptr, access_type, status);
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_write(int64_t dmi_addr, int64_t size, void *out_ptr, int8_t access_type, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(_wdmi->profile_flag == TRUE
     && profile_get_type(_wdmi->profile) == PROFILE_OUT
     && _tls_profile_flag == TRUE
     && access_type == DMI_EXCLUSIVE)
    {
      ret = profile_record_access(_wdmi->profile, dmi_addr, size);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
#if DEBUG_ACCESS
  int ran = mrand_int(0, 1);
  switch(ran)
    {
    case 0:
      spi_write_addr(_wdmi->dmi, dmi_addr, size, out_ptr, DMI_EXCLUSIVE, status);
      break;
    case 1:
      spi_write_addr(_wdmi->dmi, dmi_addr, size, out_ptr, DMI_PUT, status);
      break;
    default:
      error();
    }
#else
  spi_write_addr(_wdmi->dmi, dmi_addr, size, out_ptr, access_type, status);
#endif
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_group_write(group_t *group, void *out_ptr, int8_t access_type, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  spi_gwrite_addr(_wdmi->dmi, group, out_ptr, access_type, status);
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_watch(int64_t dmi_addr, int64_t size, void *in_ptr, void *out_ptr, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(_wdmi->profile_flag == TRUE
     && profile_get_type(_wdmi->profile) == PROFILE_OUT
     && _tls_profile_flag == TRUE)
    {
      ret = profile_record_access(_wdmi->profile, dmi_addr, size);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  spi_watch_addr(_wdmi->dmi, dmi_addr, size, in_ptr, out_ptr, status);
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_atomic(int64_t dmi_addr, int64_t size, void *out_ptr, int64_t out_size, void *in_ptr, int64_t in_size, int8_t tag, int8_t access_type, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(_wdmi->profile_flag == TRUE
     && profile_get_type(_wdmi->profile) == PROFILE_OUT
     && _tls_profile_flag == TRUE
     && access_type == DMI_EXCLUSIVE)
    {
      ret = profile_record_access(_wdmi->profile, dmi_addr, size);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  spi_atomic_addr(_wdmi->dmi, dmi_addr, size, out_ptr, out_size, in_ptr, in_size, tag, access_type, status);
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_cas(int64_t dmi_addr, int64_t size, void *cmp_ptr, void *swap_ptr, int8_t *cas_flag_ptr, int8_t access_type, status_t *status)
{
  status_t tmp_status;
  int8_t *buf;
  int32_t ret;
  int64_t uoffset, usize;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  buf = my_malloc(size + size);
  uoffset = 0;
  usize = size;
  memcpy(buf + uoffset, cmp_ptr, usize);
  uoffset += usize;
  usize = size;
  memcpy(buf + uoffset, swap_ptr, usize);
  uoffset += usize;
  
#if DEBUG_ACCESS
  int ran = mrand_int(0, 1);
  switch(ran)
    {
    case 0:
      ret = DMI_atomic(dmi_addr, size, buf, uoffset, cas_flag_ptr, sizeof(int8_t), SYS_CAS_TAG, DMI_EXCLUSIVE, status);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    case 1:
      ret = DMI_atomic(dmi_addr, size, buf, uoffset, cas_flag_ptr, sizeof(int8_t), SYS_CAS_TAG, DMI_PUT, status);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    default:
      error();
    }
#else
  ret = DMI_atomic(dmi_addr, size, buf, uoffset, cas_flag_ptr, sizeof(int8_t), SYS_CAS_TAG, access_type, status);
  throw_or_catch(_wdmi->config->catch_flag, ret);
#endif
  
  status->buf = buf;
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_fas(int64_t dmi_addr, int64_t size, void *out_ptr, void *in_ptr, int8_t access_type, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
#if DEBUG_ACCESS
  int ran = mrand_int(0, 1);
  switch(ran)
    {
    case 0:
      ret = DMI_atomic(dmi_addr, size, out_ptr, size, in_ptr, size, SYS_FAS_TAG, DEBUG_EXCLUSIVE, status);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    case 1:
      ret = DMI_atomic(dmi_addr, size, out_ptr, size, in_ptr, size, SYS_FAS_TAG, DEBUG_PUT, status);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    default:
      error();
    }
#else
  ret = DMI_atomic(dmi_addr, size, out_ptr, size, in_ptr, size, SYS_FAS_TAG, access_type, status);
  throw_or_catch(_wdmi->config->catch_flag, ret);
#endif
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_fad(int64_t dmi_addr, int64_t add_value, int64_t *fetch_value_ptr, int8_t access_type, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  int64_t fetch_size;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  if(fetch_value_ptr == NULL)
    {
      fetch_size = 0;
    }
  else
    {
      fetch_size = sizeof(int64_t);
    }
  
#if DEBUG_ACCESS
  int ran = mrand_int(0, 1);
  switch(ran)
    {
    case 0:
      ret = DMI_atomic(dmi_addr, sizeof(int64_t), &add_value, sizeof(int64_t), fetch_value_ptr, fetch_size, SYS_FAD_TAG, DMI_EXCLUSIVE, status);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    case 1:
      ret = DMI_atomic(dmi_addr, sizeof(int64_t), &add_value, sizeof(int64_t), fetch_value_ptr, fetch_size, SYS_FAD_TAG, DMI_PUT, status);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    default:
      error();
    }
#else
  ret = DMI_atomic(dmi_addr, sizeof(int64_t), &add_value, sizeof(int64_t), fetch_value_ptr, fetch_size, SYS_FAD_TAG, access_type, status);
  throw_or_catch(_wdmi->config->catch_flag, ret);
#endif
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_save(int64_t dmi_addr, int64_t size)
{
  int32_t ret;
  
  ret = spi_save_addr(_wdmi->dmi, dmi_addr, size);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_unsave(int64_t dmi_addr, int64_t size)
{
  int32_t ret;
  
  ret = spi_unsave_addr(_wdmi->dmi, dmi_addr, size);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_new_object(int32_t *object_ids, int64_t *object_sizes, int32_t object_num, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  spi_new_object(_wdmi->dmi, object_ids, object_sizes, object_num, status);
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_delete_object(int32_t *object_ids, int32_t object_num, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  spi_delete_object(_wdmi->dmi, object_ids, object_num, status);
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_read_object(int32_t object_id, int64_t object_offset, int64_t size, void *in_ptr, int8_t access_type, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
#if DEBUG_ACCESS
  int ran = mrand_int(0, 2);
  switch(ran)
    {
    case 0:
      spi_read_object(_wdmi->dmi, object_id, object_offset, size, in_ptr, DMI_GET, status);
      break;
    case 1:
      spi_read_object(_wdmi->dmi, object_id, object_offset, size, in_ptr, DMI_INVALIDATE, status);
      break;
    case 2:
      spi_read_object(_wdmi->dmi, object_id, object_offset, size, in_ptr, DMI_GET, status);
      break;
    default:
      error();
    }
#else
  spi_read_object(_wdmi->dmi, object_id, object_offset, size, in_ptr, access_type, status);
#endif
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_gread_object(int32_t object_id, int64_t *object_offsets, int64_t *sizes, int64_t *ptr_offsets, int32_t object_num, void *in_ptr, int8_t access_type, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  spi_gread_object(_wdmi->dmi, object_id, object_offsets, sizes, ptr_offsets, object_num, in_ptr, access_type, status);
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_write_object(int32_t object_id, int64_t object_offset, int64_t size, void *out_ptr, int8_t access_type, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
#if DEBUG_ACCESS
  int ran = mrand_int(0, 1);
  switch(ran)
    {
    case 0:
      spi_write_object(_wdmi->dmi, object_id, object_offset, size, out_ptr, DMI_EXCLUSIVE, status);
      break;
    case 1:
      spi_write_object(_wdmi->dmi, object_id, object_offset, size, out_ptr, DMI_PUT, status);
      break;
    default:
      error();
    }
#else
  spi_write_object(_wdmi->dmi, object_id, object_offset, size, out_ptr, access_type, status);
#endif
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_gwrite_object(int32_t object_id, int64_t *object_offsets, int64_t *sizes, int64_t *ptr_offsets, int32_t object_num, void *out_ptr, int8_t access_type, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  spi_gwrite_object(_wdmi->dmi, object_id, object_offsets, sizes, ptr_offsets, object_num, out_ptr, access_type, status);
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_watch_object(int32_t object_id, int64_t object_offset, int64_t size, void *in_ptr, void *out_ptr, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  spi_watch_object(_wdmi->dmi, object_id, object_offset, size, in_ptr, out_ptr, status);
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_atomic_object(int32_t object_id, void *out_ptr, int64_t out_size, void *in_ptr, int64_t in_size, int8_t tag, int8_t access_type, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  spi_atomic_object(_wdmi->dmi, object_id, out_ptr, out_size, in_ptr, in_size, tag, access_type, status);
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_cas_object(int32_t object_id, int64_t object_offset, int64_t size, void *cmp_ptr, void *swap_ptr, int8_t *cas_flag_ptr, int8_t access_type, status_t *status)
{
  status_t tmp_status;
  int8_t *buf;
  int32_t ret;
  int64_t uoffset, usize;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  buf = my_malloc(sizeof(int64_t) + sizeof(int64_t) + size + size);
  uoffset = 0;
  usize = sizeof(int64_t);
  memcpy(buf + uoffset, &object_offset, usize);
  uoffset += usize;
  usize = sizeof(int64_t);
  memcpy(buf + uoffset, &size, usize);
  uoffset += usize;
  usize = size;
  memcpy(buf + uoffset, cmp_ptr, usize);
  uoffset += usize;
  usize = size;
  memcpy(buf + uoffset, swap_ptr, usize);
  uoffset += usize;
  
#if DEBUG_ACCESS
  int ran = mrand_int(0, 1);
  switch(ran)
    {
    case 0:
      ret = DMI_atomic_object(object_id, status->buf, uoffset, cas_flag_ptr, sizeof(int8_t), SYS_CAS_OBJECT_TAG, access_type, status);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    case 1:
      ret = DMI_atomic_object(object_id, status->buf, uoffset, cas_flag_ptr, sizeof(int8_t), SYS_CAS_OBJECT_TAG, access_type, status);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    default:
      error();
    }
#else
  ret = DMI_atomic_object(object_id, status->buf, uoffset, cas_flag_ptr, sizeof(int8_t), SYS_CAS_OBJECT_TAG, access_type, status);
  throw_or_catch(_wdmi->config->catch_flag, ret);
#endif
  
  status->buf = buf;
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_fas_object(int32_t object_id, int64_t object_offset, int64_t size, void *out_ptr, void *in_ptr, int8_t access_type, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  int64_t uoffset, usize;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  status->buf = my_malloc(sizeof(int64_t) + sizeof(int64_t) + size);
  uoffset = 0;
  usize = sizeof(int64_t);
  memcpy(status->buf + uoffset, &object_offset, usize);
  uoffset += usize;
  usize = sizeof(int64_t);
  memcpy(status->buf + uoffset, &size, usize);
  uoffset += usize;
  usize = size;
  memcpy(status->buf + uoffset, out_ptr, usize);
  uoffset += usize;
  
#if DEBUG_ACCESS
  int ran = mrand_int(0, 1);
  switch(ran)
    {
    case 0:
      ret = DMI_atomic_object(object_id, status->buf, uoffset, in_ptr, size, SYS_FAS_OBJECT_TAG, access_type, status);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    case 1:
      ret = DMI_atomic_object(object_id, status->buf, uoffset, in_ptr, size, SYS_FAS_OBJECT_TAG, access_type, status);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    default:
      error();
    }
#else
  ret = DMI_atomic_object(object_id, status->buf, uoffset, in_ptr, size, SYS_FAS_OBJECT_TAG, access_type, status);
  throw_or_catch(_wdmi->config->catch_flag, ret);
#endif
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_getsize_object(int32_t object_id, int64_t *slide_size_ptr, int8_t access_type, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  spi_getsize_object(_wdmi->dmi, object_id, slide_size_ptr, access_type, status);
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_setsize_object(int32_t object_id, int64_t slide_size, int8_t access_type, status_t *status)
{
  status_t tmp_status;
  int32_t ret;
  
  if(status == NULL)
    {
      status = &tmp_status;
    }
  spi_start_status(status);
  
  spi_setsize_object(_wdmi->dmi, object_id, slide_size, access_type, status);
  
  if(status == &tmp_status)
    {
      spi_wait_status(status, &ret);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t DMI_save_object(int32_t object_id)
{
  int32_t ret;
  
  ret = spi_save_object(_wdmi->dmi, object_id);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_unsave_object(int32_t object_id)
{
  int32_t ret;
  
  ret = spi_unsave_object(_wdmi->dmi, object_id);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_check(status_t *status, int32_t *ret_ptr)
{
  int32_t ret;
  
  ret = spi_check_status(status, ret_ptr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

void DMI_wait(status_t *status, int32_t *ret_ptr)
{
  spi_wait_status(status, ret_ptr);
  return;
}

int32_t DMI_malloc(int64_t *addr_ptr, int64_t size, int64_t page_size)
{
  int32_t ret;
  
  ret = dmi_memalloc_malloc(_wdmi->dmi_memalloc_addr, addr_ptr, size, page_size, FALSE);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_realloc(int64_t old_addr, int64_t *new_addr_ptr, int64_t size, int64_t page_size)
{
  int32_t ret;
  
  ret = dmi_memalloc_realloc(_wdmi->dmi_memalloc_addr, old_addr, new_addr_ptr, size, page_size, FALSE);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_free(int64_t addr)
{
  int32_t ret;
  
  ret = dmi_memalloc_free(_wdmi->dmi_memalloc_addr, addr, FALSE);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

void* DMI_thread_mmap(void *start, size_t length, int prot, int flags, int fd, off_t offset)
{
  void *ptr;
  
  ptr = spi_mmap_tls(_wdmi->dmi, start, length, prot, flags, fd, offset);
  return ptr;
}

void* DMI_thread_mremap(void *old_address, size_t old_size, size_t new_size, int flags)
{
  void *ptr;
  
  ptr = spi_mremap_tls(_wdmi->dmi, old_address, old_size, new_size, flags);
  return ptr;
}

int32_t DMI_thread_mprotect(void *addr, size_t len, int prot)
{
  int32_t ret;
  
  ret = spi_mprotect_tls(_wdmi->dmi, addr, len, prot);
  return ret;
}

int32_t DMI_thread_munmap(void *start, size_t length)
{
  int32_t ret;
  
  ret = spi_munmap_tls(_wdmi->dmi, start, length);
  return ret;
}

void* DMI_thread_malloc(int64_t size)
{
  void *ptr;
  
  ptr = dmi_kr_malloc(size);
  return ptr;
}

void* DMI_thread_realloc(void *ptr, int64_t size)
{
  void *new_ptr;
  
  new_ptr = dmi_kr_realloc(ptr, size);
  return new_ptr;
}

void DMI_thread_free(void *ptr)
{
  dmi_kr_free(ptr);
  return;
}

int32_t DMI_spin_init(int64_t dmi_spinlock_addr)
{
  int32_t ret;
  
  ret = dmi_spin_init(dmi_spinlock_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_spin_destroy(int64_t dmi_spinlock_addr)
{
  int32_t ret;
  
  ret = dmi_spin_destroy(dmi_spinlock_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_spin_lock(int64_t dmi_spinlock_addr)
{
  int32_t ret;
  
  ret = dmi_spin_lock(dmi_spinlock_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_spin_unlock(int64_t dmi_spinlock_addr)
{
  int32_t ret;
  
  ret = dmi_spin_unlock(dmi_spinlock_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_spin_trylock(int64_t dmi_spinlock_addr, int32_t *try_flag_ptr)
{
  int32_t ret;
  
  ret = dmi_spin_trylock(dmi_spinlock_addr, try_flag_ptr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_mutex_init(int64_t dmi_mutex_addr)
{
  int32_t ret;
  
  ret = dmi_mutex_init(dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_mutex_destroy(int64_t dmi_mutex_addr)
{
  int32_t ret;
  
  ret = dmi_mutex_destroy(dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_mutex_lock(int64_t dmi_mutex_addr)
{
  int32_t ret;
  
  ret = dmi_mutex_lock(dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_mutex_unlock(int64_t dmi_mutex_addr)
{
  int32_t ret;
  
  ret = dmi_mutex_unlock(dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_mutex_trylock(int64_t dmi_mutex_addr, int32_t *try_flag_ptr)
{
  int32_t ret;
  
  ret = dmi_mutex_trylock(dmi_mutex_addr, try_flag_ptr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_cond_init(int64_t dmi_cond_addr)
{
  int32_t ret;
  
  ret = dmi_cond_init(dmi_cond_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_cond_destroy(int64_t dmi_cond_addr)
{
  int32_t ret;
  
  ret = dmi_cond_destroy(dmi_cond_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_cond_wait(int64_t dmi_cond_addr, int64_t dmi_mutex_addr)
{
  int32_t ret;
  
  ret = dmi_cond_wait(dmi_cond_addr, dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_cond_signal(int64_t dmi_cond_addr)
{
  int32_t ret;
  
  ret = dmi_cond_signal(dmi_cond_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_cond_broadcast(int64_t dmi_cond_addr)
{
  int32_t ret;
  
  ret = dmi_cond_broadcast(dmi_cond_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_idpool_init(int64_t dmi_idpool_addr, int32_t id_max)
{
  int32_t ret;
  
  ret = dmi_idpool_init(dmi_idpool_addr, id_max);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_idpool_destroy(int64_t dmi_idpool_addr)
{
  int32_t ret;
  
  ret = dmi_idpool_destroy(dmi_idpool_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_idpool_get(int64_t dmi_idpool_addr, int32_t *id_ptr)
{
  int32_t ret;
  
  ret = dmi_idpool_get(dmi_idpool_addr, id_ptr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_idpool_put(int64_t dmi_idpool_addr, int32_t id)
{
  int32_t ret;
  
  ret = dmi_idpool_put(dmi_idpool_addr, id);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_rwset_init(int64_t dmi_rwset_addr, int64_t element_num, int64_t element_size, int32_t rwset_num)
{
  int32_t ret;
  
  ret = dmi_rwset_init(dmi_rwset_addr, element_num, element_size, rwset_num);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_rwset_destroy(int64_t dmi_rwset_addr)
{
  int32_t ret;
  
  ret = dmi_rwset_destroy(dmi_rwset_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_rwset_decompose(int64_t dmi_rwset_addr, int32_t my_id, int64_t *write_elements, int32_t write_element_num)
{
  int32_t ret;
  
  ret = dmi_rwset_decompose(dmi_rwset_addr, my_id, write_elements, write_element_num);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_local_rwset_init(rwset_t *rwset, int64_t dmi_rwset_addr, int32_t my_id, int64_t *read_elements, int32_t read_element_num)
{
  int32_t ret;
  
  ret = rwset_init(rwset, dmi_rwset_addr, my_id, read_elements, read_element_num);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_local_rwset_destroy(rwset_t *rwset)
{
  int32_t ret;
  
  ret = rwset_destroy(rwset);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_local_rwset_write(rwset_t *rwset, void *buf, status_t *status)
{
  int32_t ret;
  
  ret = rwset_write(rwset, buf, status);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_local_rwset_read(rwset_t *rwset, void *buf, status_t *status)
{
  int32_t ret;
  
  ret = rwset_read(rwset, buf, status);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_rwset2_init(int64_t dmi_rwset2_addr, int64_t element_num, int64_t element_size, int32_t rwset2_num)
{
  int32_t ret;
  
  ret = dmi_rwset2_init(dmi_rwset2_addr, element_num, element_size, rwset2_num);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_rwset2_destroy(int64_t dmi_rwset2_addr)
{
  int32_t ret;
  
  ret = dmi_rwset2_destroy(dmi_rwset2_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_rwset2_decompose(int64_t dmi_rwset2_addr, int32_t my_id, int64_t *write_elements, int32_t write_element_num, int64_t *read_elements, int32_t read_element_num)
{
  int32_t ret;
  
  ret = dmi_rwset2_decompose(dmi_rwset2_addr, my_id, write_elements, write_element_num, read_elements, read_element_num);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_local_rwset2_init(rwset2_t *rwset2, int64_t dmi_rwset2_addr, int32_t my_id)
{
  int32_t ret;
  
  ret = rwset2_init(rwset2, dmi_rwset2_addr, my_id);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_local_rwset2_destroy(rwset2_t *rwset2)
{
  int32_t ret;
  
  ret = rwset2_destroy(rwset2);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_local_rwset2_write(rwset2_t *rwset2, void *buf, status_t *status)
{
  int32_t ret;
  
  ret = rwset2_write(rwset2, buf, status);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_local_rwset2_read(rwset2_t *rwset2, void *buf, status_t *status)
{
  int32_t ret;
  
  ret = rwset2_read(rwset2, buf, status);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_barrier_init(int64_t dmi_barrier_addr)
{
  int32_t ret;
  
  ret = dmi_barrier_init(dmi_barrier_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_barrier_destroy(int64_t dmi_barrier_addr)
{
  int32_t ret;
  
  ret = dmi_barrier_destroy(dmi_barrier_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_local_barrier_init(barrier_t *barrier, int64_t dmi_barrier_addr)
{
  int32_t ret;
  
  ret = barrier_init(barrier, dmi_barrier_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_local_barrier_destroy(barrier_t *barrier)
{
  int32_t ret;
  
  ret = barrier_destroy(barrier);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_local_barrier_sync(barrier_t *barrier, int32_t pnum)
{
  int32_t ret;
  
  ret = barrier_allreduce_int64(barrier, pnum, 0, NULL, DMI_OP_SUM);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_local_barrier_allreduce(barrier_t *barrier, int32_t pnum, void *sub_value_ptr, void *value_ptr, int8_t op_type, int8_t type_type)
{
  int32_t ret;
  int64_t sub_int64_value, int64_value;
  double sub_double_value, double_value;
  
  switch(type_type)
    {
    case DMI_TYPE_CHAR:
      sub_int64_value = (int64_t)(*(char*)sub_value_ptr);
      ret = barrier_allreduce_int64(barrier, pnum, sub_int64_value, &int64_value, op_type);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      *(char*)value_ptr = (char)int64_value;
      break;
    case DMI_TYPE_SHORT:
      sub_int64_value = (int64_t)(*(short*)sub_value_ptr);
      ret = barrier_allreduce_int64(barrier, pnum, sub_int64_value, &int64_value, op_type);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      *(short*)value_ptr = (short)int64_value;
      break;
    case DMI_TYPE_INT:
      sub_int64_value = (int64_t)(*(int*)sub_value_ptr);
      ret = barrier_allreduce_int64(barrier, pnum, sub_int64_value, &int64_value, op_type);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      *(int*)value_ptr = (int)int64_value;
      break;
    case DMI_TYPE_LONG:
      sub_int64_value = (int64_t)(*(long*)sub_value_ptr);
      ret = barrier_allreduce_int64(barrier, pnum, sub_int64_value, &int64_value, op_type);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      *(long*)value_ptr = (long)int64_value;
      break;
    case DMI_TYPE_LONGLONG:
      sub_int64_value = (int64_t)(*(long long*)sub_value_ptr);
      ret = barrier_allreduce_int64(barrier, pnum, sub_int64_value, &int64_value, op_type);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      *(long long*)value_ptr = (long long)int64_value;
      break;
    case DMI_TYPE_FLOAT:
      sub_double_value = (double)(*(float*)sub_value_ptr);
      ret = barrier_allreduce_double(barrier, pnum, sub_double_value, &double_value, op_type);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      *(float*)value_ptr = (float)double_value;
      break;
    case DMI_TYPE_DOUBLE:
      sub_double_value = *(double*)sub_value_ptr;
      ret = barrier_allreduce_double(barrier, pnum, sub_double_value, &double_value, op_type);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      *(double*)value_ptr = double_value;
      break;
    default:
      error();
    }
  return TRUE;
}

int32_t DMI_welcome(int32_t dmi_id)
{
  int32_t ret;
  
  ret = dmi_member_welcome(_wdmi->dmi_member_addr, dmi_id);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_goodbye(int32_t dmi_id)
{
  int32_t ret;
  
  ret = dmi_member_goodbye(_wdmi->dmi_member_addr, dmi_id);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_poll(dmi_node_t *dmi_node_ptr)
{
  int32_t ret;
  
  ret = dmi_member_poll(_wdmi->dmi_member_addr, dmi_node_ptr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_peek(dmi_node_t *dmi_node_ptr, int8_t *flag_ptr)
{
  int32_t ret;
  
  ret = dmi_member_peek(_wdmi->dmi_member_addr, dmi_node_ptr, flag_ptr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_nodes(dmi_node_t *dmi_node_array, int32_t *num_ptr, int32_t capacity)
{
  int32_t ret;
  
  ret = dmi_member_nodes(_wdmi->dmi_member_addr, dmi_node_array, num_ptr, capacity);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_rescale(int64_t scaleunit_addr, int32_t init_node_num, int32_t thread_num)
{
  int32_t ret;
  
  ret = dmi_rescale_scale(scaleunit_addr, init_node_num, thread_num);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_check_rescale(int32_t *judge_ptr)
{
  int32_t ret;
  
  ret = dmi_rescale_check(judge_ptr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_scheduler_init(int64_t dmi_scheduler_addr)
{
  int32_t ret;
  
  ret = dmi_scheduler_init(dmi_scheduler_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_scheduler_destroy(int64_t dmi_scheduler_addr)
{
  int32_t ret;
  
  ret = dmi_scheduler_destroy(dmi_scheduler_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_scheduler_create(int64_t dmi_scheduler_addr, int32_t *sthread_id_ptr, int64_t dmi_addr)
{
  int32_t ret;
  
  ret = dmi_scheduler_create(dmi_scheduler_addr, sthread_id_ptr, dmi_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_scheduler_detach(int64_t dmi_scheduler_addr, int32_t sthread_id)
{
  int32_t ret;
  
  ret = dmi_scheduler_detach(dmi_scheduler_addr, sthread_id);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t DMI_scheduler_join(int64_t dmi_scheduler_addr, int32_t sthread_id, int64_t *dmi_addr_ptr)
{
  int32_t ret;
  
  ret = dmi_scheduler_join(dmi_scheduler_addr, sthread_id, dmi_addr_ptr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

/* rescale */

int32_t dmi_rescale_scale(int64_t scaleunit_addr, int32_t init_node_num, int32_t thread_num)
{
  barrier_t barrier;
  dmi_node_t *dmi_nodes, *in_dmi_nodes, *out_dmi_nodes;
  dmi_athread_t tmp_athread;
  dmi_athread_t *dmi_athreads;
  idpool_t *idpool;
  rescale_t rescale;
  int8_t exit_flag, flag;
  int8_t *flags;
  int32_t rank, pnum, tmp_rank, dmi_node_id, dmi_node_num, ret, athread_id, 
    in_node_num, out_node_num, node_num, i, welcome_pnum, old_pnum, judge, athread_num;
  int32_t *welcome_ranks, *ranks;
  int64_t dmi_barrier_addr, dmi_barrier2_addr, ranks_addr, flags_addr, pnum_addr,
    exit_flag_addr, rescale_addr, judge_addr;
  
  dmi_nodes = (dmi_node_t*)my_malloc(sizeof(dmi_node_t) * CONST_NODE_MAX);
  in_dmi_nodes = (dmi_node_t*)my_malloc(sizeof(dmi_node_t) * CONST_NODE_MAX);
  out_dmi_nodes = (dmi_node_t*)my_malloc(sizeof(dmi_node_t) * CONST_NODE_MAX);
  welcome_ranks = (int32_t*)my_malloc(sizeof(int32_t) * CONST_ATHREAD_MAX);
  dmi_athreads = (dmi_athread_t*)my_malloc(sizeof(dmi_athread_t) * CONST_ATHREAD_MAX);
  idpool = idpool_alloc(CONTAINER_DEFAULT);
  
  while(1)
    {
      in_node_num = 0;
      ret = DMI_nodes(dmi_nodes, &dmi_node_num, CONST_NODE_MAX);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      for(dmi_node_id = 0; dmi_node_id < dmi_node_num; dmi_node_id++)
        {
          if(dmi_nodes[dmi_node_id].state == DMI_OPEN)
            {
              in_dmi_nodes[in_node_num] = dmi_nodes[dmi_node_id];
              in_node_num++;
            }
        }
      if(DMI_start_scale)
        {
          ret = DMI_start_scale(in_dmi_nodes, in_node_num);
          if(ret > 0)
            {
              break;
            }
        }
      else
        {
          if(in_node_num >= init_node_num)
            {
              break;
            }
        }
      halt(CONST_RESCALE_HALT_INTERVAL);
    }
  
  ret = DMI_mmap(&dmi_barrier_addr, sizeof(dmi_barrier_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&dmi_barrier2_addr, sizeof(dmi_barrier_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&ranks_addr, sizeof(int32_t) * CONST_ATHREAD_MAX, 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&flags_addr, sizeof(int8_t) * CONST_ATHREAD_MAX, 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&pnum_addr, sizeof(int32_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&rescale_addr, sizeof(rescale_t) * CONST_ATHREAD_MAX, 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&exit_flag_addr, sizeof(int8_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&judge_addr, sizeof(int32_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_barrier_init(dmi_barrier_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_local_barrier_init(&barrier, dmi_barrier_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_barrier_init(dmi_barrier2_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  flags = (int8_t*)my_malloc(sizeof(int8_t) * CONST_ATHREAD_MAX);
  for(i = 0; i < CONST_ATHREAD_MAX; i++)
    {
      flags[i] = FALSE;
    }
  ret = DMI_write(flags_addr, sizeof(int8_t) * CONST_ATHREAD_MAX, flags, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  my_free(flags);
  
  ranks = (int32_t*)my_malloc(sizeof(int32_t) * CONST_ATHREAD_MAX);
  for(i = 0; i < CONST_ATHREAD_MAX; i++)
    {
      ranks[i] = i;
    }
  ret = DMI_write(ranks_addr, sizeof(int32_t) * CONST_ATHREAD_MAX, ranks, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  my_free(ranks);
  
  exit_flag = FALSE;
  ret = DMI_write(exit_flag_addr, sizeof(int8_t), &exit_flag, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  judge = 1;
  ret = DMI_write(judge_addr, sizeof(int32_t), &judge, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  for(rank = 0; rank < CONST_ATHREAD_MAX; rank++)
    {
      dmi_athreads[rank].state = STATE_CLOSED;
    }
  
  pnum = 0;
  node_num = 0;
  while(1)
    {
      old_pnum = pnum;
      in_node_num = 0;
      out_node_num = 0;
      
      if(judge == 1)
        {
          ret = DMI_nodes(dmi_nodes, &dmi_node_num, CONST_NODE_MAX);
          throw_or_catch(_wdmi->config->catch_flag, ret);
          
          for(dmi_node_id = 0; dmi_node_id < dmi_node_num; dmi_node_id++)
            {
              if(dmi_nodes[dmi_node_id].state == DMI_OPEN)
                {
                  if(thread_num <= 0)
                    {
                      pnum += dmi_nodes[dmi_node_id].core;
                    }
                  else
                    {
                      pnum += thread_num;
                    }
                  in_dmi_nodes[in_node_num++] = dmi_nodes[dmi_node_id];
                }
              else if(dmi_nodes[dmi_node_id].state == DMI_CLOSE)
                {
                  if(thread_num <= 0)
                    {
                      pnum -= dmi_nodes[dmi_node_id].core;
                    }
                  else
                    {
                      pnum -= thread_num;
                    }
                  out_dmi_nodes[out_node_num++] = dmi_nodes[dmi_node_id];
                  
                  for(rank = 0; rank < CONST_ATHREAD_MAX; rank++)
                    {
                      if(dmi_athreads[rank].state == STATE_OPENED
                         && dmi_athreads[rank].dmi_thread.dmi_id == dmi_nodes[dmi_node_id].dmi_id)
                        {
                          dmi_athreads[rank].state = STATE_CLOSE;
                          
                          flag = TRUE;
                          ret = DMI_write(flags_addr + dmi_athreads[rank].athread_id * sizeof(int8_t), sizeof(int8_t), &flag, DMI_EXCLUSIVE, NULL);
                          throw_or_catch(_wdmi->config->catch_flag, ret);
                        }
                    }
                }
            }
          
          if(pnum <= 0)
            {
              pnum = old_pnum;
              halt(CONST_RESCALE_HALT_INTERVAL);
              continue;
            }
          
          welcome_pnum = 0;
          for(rank = 0; rank < CONST_ATHREAD_MAX; rank++)
            {
              if(dmi_athreads[rank].state == STATE_OPENED)
                {
                  welcome_ranks[welcome_pnum++] = rank;
                }
            }
          for(rank = 0; rank < welcome_pnum; rank++)
            {
              tmp_athread = dmi_athreads[rank];
              dmi_athreads[rank] = dmi_athreads[welcome_ranks[rank]];
              dmi_athreads[welcome_ranks[rank]] = tmp_athread;
            }
          rank = welcome_pnum;
          tmp_rank = pnum;
          for(i = 0; i < in_node_num; i++)
            {
              if(thread_num <= 0)
                {
                  athread_num = in_dmi_nodes[i].core;
                }
              else
                {
                  athread_num = thread_num;
                }
              for(athread_id = 0; athread_id < athread_num; athread_id++)
                {
                  tmp_athread = dmi_athreads[rank];
                  dmi_athreads[rank] = dmi_athreads[tmp_rank];
                  dmi_athreads[tmp_rank] = tmp_athread;
                  tmp_rank++;
                  
                  dmi_athreads[rank].state = STATE_OPEN;
                  dmi_athreads[rank].athread_id = idpool_get(idpool);
                  dmi_athreads[rank].dmi_id = in_dmi_nodes[i].dmi_id;
                  rank++;
                }
            }
          fprintf(stdout, "rank=%d pnum=%d\n", rank, pnum);
          if(rank != pnum) error();
          
          ret = DMI_write(pnum_addr, sizeof(int32_t), &pnum, DMI_EXCLUSIVE, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
        }
      
      //outn("C_scale before pnum=%d", old_pnum + 1);
      ret = DMI_local_barrier_sync(&barrier, old_pnum + 1); /* (C) */
      throw_or_catch(_wdmi->config->catch_flag, ret);
      //outn("C_scale after pnum=%d", old_pnum + 1);
      
      for(rank = 0; rank < pnum; rank++)
        {
          ret = DMI_write(ranks_addr + dmi_athreads[rank].athread_id * sizeof(int32_t), sizeof(int32_t), &rank, DMI_EXCLUSIVE, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
        }
      
      for(i = 0; i < in_node_num; i++)
        {
          node_num++;
          ret = DMI_welcome(in_dmi_nodes[i].dmi_id);
          throw_or_catch(_wdmi->config->catch_flag, ret);
        }
      for(rank = 0; rank < CONST_ATHREAD_MAX; rank++)
        {
          if(dmi_athreads[rank].state == STATE_OPEN)
            {
              dmi_athreads[rank].state = STATE_OPENED;
              
              rescale.athread_id = dmi_athreads[rank].athread_id;
              rescale.ranks_addr = ranks_addr;
              rescale.flags_addr = flags_addr;
              rescale.pnum_addr = pnum_addr;
              rescale.exit_flag_addr = exit_flag_addr;
              rescale.judge_addr = judge_addr;
              rescale.scaleunit_addr = scaleunit_addr;
              rescale.dmi_barrier_addr = dmi_barrier_addr;
              rescale.dmi_barrier2_addr = dmi_barrier2_addr;
              
              ret = DMI_write(rescale_addr + dmi_athreads[rank].athread_id * sizeof(rescale_t), sizeof(rescale_t), &rescale, DMI_EXCLUSIVE, NULL);
              throw_or_catch(_wdmi->config->catch_flag, ret);
              
              ret = DMI_create2(&dmi_athreads[rank].dmi_thread, dmi_athreads[rank].dmi_id, rescale_addr + dmi_athreads[rank].athread_id * sizeof(rescale_t), 0, FALSE, THREAD_SCALEUNIT, 0, 0, NULL);
              throw_or_catch(_wdmi->config->catch_flag, ret);
            }
        }
      
      //outn("A_scale before pnum=%d", pnum + 1);
      ret = DMI_local_barrier_sync(&barrier, pnum + 1); /* (A) */
      throw_or_catch(_wdmi->config->catch_flag, ret);
      //outn("A_scale after pnum=%d", pnum + 1);
      
      for(rank = 0; rank < CONST_ATHREAD_MAX; rank++)
        {
          if(dmi_athreads[rank].state == STATE_CLOSE)
            {
              dmi_athreads[rank].state = STATE_CLOSED;
              
              ret = DMI_join(dmi_athreads[rank].dmi_thread, NULL, NULL);
              throw_or_catch(_wdmi->config->catch_flag, ret);
              
              flag = FALSE;
              ret = DMI_write(flags_addr + dmi_athreads[rank].athread_id * sizeof(int8_t), sizeof(int8_t), &flag, DMI_EXCLUSIVE, NULL);
              throw_or_catch(_wdmi->config->catch_flag, ret);
              
              idpool_put(idpool, dmi_athreads[rank].athread_id);
            }
        }
      for(i = 0; i < out_node_num; i++)
        {
          node_num--;
          ret = DMI_goodbye(out_dmi_nodes[i].dmi_id);
          throw_or_catch(_wdmi->config->catch_flag, ret);
        }
      
      //outn("B_scale before pnum=%d", pnum + 1);
      ret = DMI_local_barrier_sync(&barrier, pnum + 1); /* (B) */
      throw_or_catch(_wdmi->config->catch_flag, ret);
      //outn("B_scale after pnum=%d", pnum + 1);
      
      ret = DMI_read(exit_flag_addr, sizeof(int8_t), &exit_flag, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      if(exit_flag == TRUE)
        {
          break;
        }
      
      ret = DMI_read(judge_addr, sizeof(int32_t), &judge, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  for(rank = 0; rank < CONST_ATHREAD_MAX; rank++)
    {
      if(dmi_athreads[rank].state == STATE_OPENED)
        {
          dmi_athreads[rank].state = STATE_CLOSE;
          
          flag = TRUE;
          ret = DMI_write(flags_addr + dmi_athreads[rank].athread_id * sizeof(int8_t), sizeof(int8_t), &flag, DMI_EXCLUSIVE, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
        }
    }
  
  //outn("C2_scale before pnum=%d", pnum + 1);
  ret = DMI_local_barrier_sync(&barrier, pnum + 1); /* (C) */
  throw_or_catch(_wdmi->config->catch_flag, ret);
  //outn("C2_scale after pnum=%d", pnum + 1);
  
  for(rank = 0; rank < CONST_ATHREAD_MAX; rank++)
    {
      if(dmi_athreads[rank].state == STATE_CLOSE)
        {
          dmi_athreads[rank].state = STATE_CLOSED;
          
          ret = DMI_join(dmi_athreads[rank].dmi_thread, NULL, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
          
          flag = FALSE;
          ret = DMI_write(flags_addr + dmi_athreads[rank].athread_id * sizeof(int8_t), sizeof(int8_t), &flag, DMI_EXCLUSIVE, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
          
          idpool_put(idpool, dmi_athreads[rank].athread_id);
        }
    }
  
  dmi_member_rescalefad(_wdmi->dmi_member_addr, node_num, NULL);
  catch(ret);
  
  ret = DMI_barrier_destroy(dmi_barrier2_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_local_barrier_destroy(&barrier);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_barrier_destroy(dmi_barrier_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_munmap(judge_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(exit_flag_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(rescale_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(pnum_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(flags_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(ranks_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_barrier2_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_barrier_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  idpool_free(idpool);
  my_free(dmi_athreads);
  my_free(welcome_ranks);
  my_free(out_dmi_nodes);
  my_free(in_dmi_nodes);
  my_free(dmi_nodes);
  return TRUE;
}

int32_t dmi_rescale_main(int64_t rescale_addr)
{
  scale_thread_t *scale_thread;
  rescale_t rescale;
  barrier_t barrier, barrier2;
  int8_t flag, exit_flag;
  int32_t ret, my_rank, pnum, ret_code, judge;
  
  ret = DMI_read(rescale_addr, sizeof(rescale_t), &rescale, DMI_GET, NULL);
  catch(ret);
  
  ret = DMI_local_barrier_init(&barrier, rescale.dmi_barrier_addr);
  catch(ret);
  ret = DMI_local_barrier_init(&barrier2, rescale.dmi_barrier2_addr);
  catch(ret);
  
  while(1)
    {
      ret = DMI_read(rescale.pnum_addr, sizeof(int32_t), &pnum, DMI_GET, NULL);
      catch(ret);
      
      //outn("A_main before pnum=%d", pnum + 1);
      ret = DMI_local_barrier_sync(&barrier, pnum + 1); /* (A) */
      catch(ret);
      //outn("A_main after pnum=%d", pnum + 1);
      
      ret = DMI_read(rescale.ranks_addr + rescale.athread_id * sizeof(int32_t), sizeof(int32_t), &my_rank, DMI_GET, NULL);
      catch(ret);
      
      scale_thread = scale_thread_alloc();
      scale_thread->my_rank = my_rank;
      scale_thread->pnum = pnum;
      scale_thread->barrier = &barrier2;
      _tls_wdmi_thread->scale_thread = scale_thread;
      
      if(DMI_scaleunit)
        {
          if(my_rank == 0)
            {
              //outn("reconfigure finish time=%lf", get_time());
            }
          ret_code = DMI_scaleunit(my_rank, pnum, rescale.scaleunit_addr);
          if(my_rank == 0)
            {
              //outn("reconfigure start time=%lf", get_time());
            }
          
          if(my_rank == 0)
            {
              if(ret_code == 0)
                {
                  exit_flag = TRUE;
                  ret = DMI_write(rescale.exit_flag_addr, sizeof(int8_t), &exit_flag, DMI_PUT, NULL);
                  catch(ret);
                }
              
              judge = sub_judge_rescale();
              ret = DMI_write(rescale.judge_addr, sizeof(int32_t), &judge, DMI_PUT, NULL);
              catch(ret);
            }
        }
      else
        {
          fprintf(stderr, "error : DMI_scaleunit is not defined!\n");
          error();
        }
      
      scale_thread = _tls_wdmi_thread->scale_thread;
      scale_thread_free(scale_thread);
      
      //outn("B_main before pnum=%d", pnum + 1);
      ret = DMI_local_barrier_sync(&barrier, pnum + 1); /* (B) */
      catch(ret);
      //outn("B_main after pnum=%d", pnum + 1);
      
      //outn("C_main before pnum=%d", pnum + 1);
      ret = DMI_local_barrier_sync(&barrier, pnum + 1); /* (C) */
      catch(ret);
      //outn("C_main after pnum=%d", pnum + 1);
      
      ret = DMI_read(rescale.flags_addr + rescale.athread_id * sizeof(int8_t), sizeof(int8_t), &flag, DMI_GET, NULL);
      catch(ret);
      if(flag == TRUE)
        {
          break;
        }
    }
  
  ret = DMI_local_barrier_destroy(&barrier2);
  catch(ret);
  ret = DMI_local_barrier_destroy(&barrier);
  catch(ret);
  return DMI_NULL;
}

int32_t sub_judge_rescale(void)
{
  dmi_node_t *in_dmi_nodes, *out_dmi_nodes, *cur_dmi_nodes, *dmi_nodes;
  int32_t in_node_num, out_node_num, cur_node_num, dmi_node_num, dmi_node_id, judge, ret;
  
  dmi_nodes = (dmi_node_t*)my_malloc(sizeof(dmi_node_t) * CONST_NODE_MAX);
  in_dmi_nodes = (dmi_node_t*)my_malloc(sizeof(dmi_node_t) * CONST_NODE_MAX);
  out_dmi_nodes = (dmi_node_t*)my_malloc(sizeof(dmi_node_t) * CONST_NODE_MAX);
  cur_dmi_nodes = (dmi_node_t*)my_malloc(sizeof(dmi_node_t) * CONST_NODE_MAX);
  
  in_node_num = 0;
  out_node_num = 0;
  cur_node_num = 0;
  ret = DMI_nodes(dmi_nodes, &dmi_node_num, CONST_NODE_MAX);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  for(dmi_node_id = 0; dmi_node_id < dmi_node_num; dmi_node_id++)
    {
      if(dmi_nodes[dmi_node_id].state == DMI_OPEN)
        {
          in_dmi_nodes[in_node_num++] = dmi_nodes[dmi_node_id];
        }
      else if(dmi_nodes[dmi_node_id].state == DMI_CLOSE)
        {
          out_dmi_nodes[out_node_num++] = dmi_nodes[dmi_node_id];
        }
      else if(dmi_nodes[dmi_node_id].state == DMI_OPENED)
        {
          cur_dmi_nodes[cur_node_num++] = dmi_nodes[dmi_node_id];
        }
    }
  
  if(DMI_judge_rescale)
    {
      judge = DMI_judge_rescale(in_dmi_nodes, out_dmi_nodes, cur_dmi_nodes, in_node_num, out_node_num, cur_node_num);
    }
  else
    {
      judge = 0;
    }
  
  my_free(cur_dmi_nodes);
  my_free(out_dmi_nodes);
  my_free(in_dmi_nodes);
  my_free(dmi_nodes);
  return judge;
}

int32_t dmi_rescale_check(int32_t *judge_ptr)
{
  scale_thread_t *scale_thread;
  int32_t sub_judge, ret;
  
  scale_thread = _tls_wdmi_thread->scale_thread;
  
  if(scale_thread->my_rank == 0)
    {
      sub_judge = sub_judge_rescale();
    }
  else
    {
      sub_judge = 0;
    }
  ret = DMI_local_barrier_allreduce(scale_thread->barrier, scale_thread->pnum, &sub_judge, judge_ptr, DMI_OP_SUM, DMI_TYPE_INT);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

/* scheduler */

int32_t dmi_scheduler_init(int64_t dmi_scheduler_addr)
{
  dmi_scheduler_t dmi_scheduler;
  resource_t *resources;
  sthread_t *sthreads;
  int8_t init_flag, final_flag;
  int8_t *phases;
  int32_t ret, sthread_id, dmi_id;
  
  ret = DMI_rank(&dmi_scheduler.dmi_id);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_welcome(dmi_scheduler.dmi_id);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mmap(&dmi_scheduler.sthreads_addr, sizeof(sthread_t) * CONST_STHREAD_MAX, 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&dmi_scheduler.resources_addr, sizeof(resource_t) * CONST_NODE_MAX, 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&dmi_scheduler.phases_addr, sizeof(int8_t) * CONST_STHREAD_MAX, 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&dmi_scheduler.init_flag_addr, sizeof(int8_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&dmi_scheduler.final_flag_addr, sizeof(int8_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&dmi_scheduler.dmi_cond_addr, sizeof(dmi_cond_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&dmi_scheduler.dmi_mutex_addr, sizeof(dmi_mutex_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&dmi_scheduler.dmi_mutex2_addr, sizeof(dmi_mutex_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&dmi_scheduler.dmi_idpool_addr, sizeof(dmi_idpool_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  init_flag = FALSE;
  ret = DMI_write(dmi_scheduler.init_flag_addr, sizeof(int8_t), &init_flag, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  final_flag = FALSE;
  ret = DMI_write(dmi_scheduler.final_flag_addr, sizeof(int8_t), &final_flag, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  resources = (resource_t*)my_malloc(sizeof(resource_t) * CONST_NODE_MAX);
  for(dmi_id = 0; dmi_id < CONST_NODE_MAX; dmi_id++)
    {
      resources[dmi_id].state = STATE_CLOSE;
      resources[dmi_id].sthread_num = 0;
    }
  ret = DMI_write(dmi_scheduler.resources_addr, sizeof(resource_t) * CONST_NODE_MAX, resources, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  my_free(resources);
  
  sthreads = (sthread_t*)my_malloc(sizeof(sthread_t) * CONST_STHREAD_MAX);
  for(sthread_id = 0; sthread_id < CONST_STHREAD_MAX; sthread_id++)
    {
      sthreads[sthread_id].state = STATE_EXIT;
      sthreads[sthread_id].dmi_addr = DMI_NULL;
      sthreads[sthread_id].dmi_thread.dmi_id = SYS_ID_UNDEF;
      sthreads[sthread_id].dmi_thread.thread_id = SYS_ID_UNDEF;
    }
  ret = DMI_write(dmi_scheduler.sthreads_addr, sizeof(sthread_t) * CONST_STHREAD_MAX, sthreads, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  my_free(sthreads);
  
  phases = (int8_t*)my_malloc(sizeof(int8_t) * CONST_STHREAD_MAX);
  for(sthread_id = 0; sthread_id < CONST_STHREAD_MAX; sthread_id++)
    {
      phases[sthread_id] = 0;
    }
  ret = DMI_write(dmi_scheduler.phases_addr, sizeof(int8_t) * CONST_STHREAD_MAX, phases, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  my_free(phases);
  
  ret = DMI_mutex_init(dmi_scheduler.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mutex_init(dmi_scheduler.dmi_mutex2_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_cond_init(dmi_scheduler.dmi_cond_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_idpool_init(dmi_scheduler.dmi_idpool_addr, CONST_STHREAD_MAX);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_write(dmi_scheduler_addr, sizeof(dmi_scheduler_t), &dmi_scheduler, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
    
  ret = DMI_create2(&dmi_scheduler.dmi_thread, dmi_scheduler.dmi_id, dmi_scheduler_addr, 0, TRUE, THREAD_SCHEDULER_A, 0, 0, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_write(dmi_scheduler_addr, sizeof(dmi_scheduler_t), &dmi_scheduler, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_lock(dmi_scheduler.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  while(1)
    {
      ret = DMI_read(dmi_scheduler.init_flag_addr, sizeof(int8_t), &init_flag, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      if(init_flag == TRUE)
        {
          break;
        }
      
      ret = DMI_cond_wait(dmi_scheduler.dmi_cond_addr, dmi_scheduler.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  ret = DMI_mutex_unlock(dmi_scheduler.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_scheduler_destroy(int64_t dmi_scheduler_addr)
{
  dmi_scheduler_t dmi_scheduler;
  int8_t final_flag;
  int32_t ret;
  
  ret = DMI_read(dmi_scheduler_addr, sizeof(dmi_scheduler_t), &dmi_scheduler, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  final_flag = TRUE;
  ret = DMI_write(dmi_scheduler.final_flag_addr, sizeof(int8_t), &final_flag, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_join(dmi_scheduler.dmi_thread, NULL, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_destroy(dmi_scheduler.dmi_mutex2_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mutex_destroy(dmi_scheduler.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_cond_destroy(dmi_scheduler.dmi_cond_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_idpool_destroy(dmi_scheduler.dmi_idpool_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_munmap(dmi_scheduler.sthreads_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_scheduler.resources_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_scheduler.phases_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_scheduler.final_flag_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_scheduler.init_flag_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_scheduler.dmi_cond_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_scheduler.dmi_mutex2_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_scheduler.dmi_mutex_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_scheduler.dmi_idpool_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_goodbye(dmi_scheduler.dmi_id);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_scheduler_create(int64_t dmi_scheduler_addr, int32_t *sthread_id_ptr, int64_t dmi_addr)
{
  dmi_scheduler_t dmi_scheduler;
  resource_t *resources;
  sthread_t sthread;
  int8_t phase;
  int32_t sthread_id, min_sthread_num, min_dmi_id, dmi_id, ret;
  
  ret = DMI_read(dmi_scheduler_addr, sizeof(dmi_scheduler_t), &dmi_scheduler, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  resources = (resource_t*)my_malloc(sizeof(resource_t) * CONST_NODE_MAX);
  
  ret = DMI_idpool_get(dmi_scheduler.dmi_idpool_addr, &sthread_id);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_lock(dmi_scheduler.dmi_mutex2_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  phase = 0;
  ret = DMI_write(dmi_scheduler.phases_addr + sthread_id * sizeof(int8_t), sizeof(int8_t), &phase, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
    
  ret = DMI_mutex_unlock(dmi_scheduler.dmi_mutex2_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_lock(dmi_scheduler.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_read(dmi_scheduler.resources_addr, sizeof(resource_t) * CONST_NODE_MAX, resources, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  min_sthread_num = CONST_STHREAD_MAX + 1;
  min_dmi_id = SYS_ID_UNDEF;
  for(dmi_id = 0; dmi_id < CONST_NODE_MAX; dmi_id++)
    {
      if(resources[dmi_id].state == STATE_OPEN && resources[dmi_id].sthread_num < min_sthread_num)
        {
          min_sthread_num = resources[dmi_id].sthread_num;
          min_dmi_id = dmi_id;
        }
    }
  if(min_dmi_id == SYS_ID_UNDEF)
    {
      throw_or_catch(_wdmi->config->catch_flag, FALSE);
    }
  resources[min_dmi_id].sthread_num++;
  
  sthread.state = STATE_CREATE;
  sthread.dmi_addr = DMI_NULL;
  
  ret = DMI_create2(&sthread.dmi_thread, min_dmi_id, dmi_scheduler_addr, 0, FALSE, THREAD_SCHEDULED, sthread_id, dmi_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_detach(sthread.dmi_thread, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_write(dmi_scheduler.resources_addr + min_dmi_id * sizeof(resource_t), sizeof(resource_t), &resources[min_dmi_id], DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_write(dmi_scheduler.sthreads_addr + sthread_id * sizeof(sthread_t), sizeof(sthread_t), &sthread, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_unlock(dmi_scheduler.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  my_free(resources);
  
  *sthread_id_ptr = sthread_id;
  return TRUE;
}

int32_t dmi_scheduler_detach(int64_t dmi_scheduler_addr, int32_t sthread_id)
{
  dmi_scheduler_t dmi_scheduler;
  resource_t resource;
  sthread_t sthread;
  int32_t ret;
  
  ret = DMI_read(dmi_scheduler_addr, sizeof(dmi_scheduler_t), &dmi_scheduler, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_lock(dmi_scheduler.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_read(dmi_scheduler.sthreads_addr + sthread_id * sizeof(sthread_t), sizeof(sthread_t), &sthread, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  switch(sthread.state)
    {
    case STATE_CREATE:
      sthread.state = STATE_DETACH;
      
      ret = DMI_write(dmi_scheduler.sthreads_addr + sthread_id * sizeof(sthread_t), sizeof(sthread_t), &sthread, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_mutex_unlock(dmi_scheduler.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    case STATE_EXIT:
      ret = DMI_read(dmi_scheduler.resources_addr + sthread.dmi_thread.dmi_id * sizeof(resource_t), sizeof(resource_t), &resource, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      resource.sthread_num--;
      
      ret = DMI_write(dmi_scheduler.resources_addr + sthread.dmi_thread.dmi_id * sizeof(resource_t), sizeof(resource_t), &resource, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
            
      ret = DMI_mutex_unlock(dmi_scheduler.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_idpool_put(dmi_scheduler.dmi_idpool_addr, sthread_id);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    case STATE_DETACH:
    case STATE_JOIN:
      ret = DMI_mutex_unlock(dmi_scheduler.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      return FALSE;
      break;
    default:
      error();
    }
  return TRUE;
}

int32_t dmi_scheduler_join(int64_t dmi_scheduler_addr, int32_t sthread_id, int64_t *dmi_addr_ptr)
{
  dmi_scheduler_t dmi_scheduler;
  resource_t resource;
  sthread_t sthread;
  int32_t ret;
  
  ret = DMI_read(dmi_scheduler_addr, sizeof(dmi_scheduler_t), &dmi_scheduler, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_lock(dmi_scheduler.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_read(dmi_scheduler.sthreads_addr + sthread_id * sizeof(sthread_t), sizeof(sthread_t), &sthread, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  switch(sthread.state)
    {
    case STATE_CREATE:
      sthread.state = STATE_JOIN;
      
      ret = DMI_write(dmi_scheduler.sthreads_addr + sthread_id * sizeof(sthread_t), sizeof(sthread_t), &sthread, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      while(1)
        {
          ret = DMI_cond_wait(dmi_scheduler.dmi_cond_addr, dmi_scheduler.dmi_mutex_addr);
          throw_or_catch(_wdmi->config->catch_flag, ret);
          
          ret = DMI_read(dmi_scheduler.sthreads_addr + sthread_id * sizeof(sthread_t), sizeof(sthread_t), &sthread, DMI_GET, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
          
          if(sthread.state == STATE_EXIT)
            {
              break;
            }
        }
      
      ret = DMI_read(dmi_scheduler.resources_addr + sthread.dmi_thread.dmi_id * sizeof(resource_t), sizeof(resource_t), &resource, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      resource.sthread_num--;
      
      ret = DMI_write(dmi_scheduler.resources_addr + sthread.dmi_thread.dmi_id * sizeof(resource_t), sizeof(resource_t), &resource, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_mutex_unlock(dmi_scheduler.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_idpool_put(dmi_scheduler.dmi_idpool_addr, sthread_id);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      if(dmi_addr_ptr != NULL)
        {
          *dmi_addr_ptr = sthread.dmi_addr;
        }
      break;
    case STATE_EXIT:
      ret = DMI_read(dmi_scheduler.resources_addr + sthread.dmi_thread.dmi_id * sizeof(resource_t), sizeof(resource_t), &resource, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      resource.sthread_num--;
      
      ret = DMI_write(dmi_scheduler.resources_addr + sthread.dmi_thread.dmi_id * sizeof(resource_t), sizeof(resource_t), &resource, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_mutex_unlock(dmi_scheduler.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_idpool_put(dmi_scheduler.dmi_idpool_addr, sthread_id);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      if(dmi_addr_ptr != NULL)
        {
          *dmi_addr_ptr = sthread.dmi_addr;
        }
      break;
    case STATE_DETACH:
    case STATE_JOIN:
      ret = DMI_mutex_unlock(dmi_scheduler.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      return FALSE;
      break;
    default:
      error();
    }
  return TRUE;
}

int32_t dmi_scheduler_main(int64_t dmi_scheduler_addr, int32_t sthread_id, int64_t dmi_addr)
{
  dmi_scheduler_t dmi_scheduler;
  resource_t resource;
  sthread_t sthread;
  int8_t phase;
  int32_t ret;
  int64_t ret_dmi_addr;
  
  ret = DMI_read(dmi_scheduler_addr, sizeof(dmi_scheduler_t), &dmi_scheduler, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret_dmi_addr = DMI_NULL;
  if(DMI_thread)
    {
      ret_dmi_addr = DMI_thread(dmi_addr);
    }
  else
    {
      fprintf(stderr, "error : DMI_thread is not defined!\n");
      error();
    }
  
  ret = DMI_mutex_lock(dmi_scheduler.dmi_mutex2_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  phase = 1;
  ret = DMI_write(dmi_scheduler.phases_addr + sthread_id * sizeof(int8_t), sizeof(int8_t), &phase, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_unlock(dmi_scheduler.dmi_mutex2_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_yield();
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_lock(dmi_scheduler.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_read(dmi_scheduler.sthreads_addr + sthread_id * sizeof(sthread_t), sizeof(sthread_t), &sthread, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  switch(sthread.state)
    {
    case STATE_CREATE:
      sthread.state = STATE_EXIT;
      sthread.dmi_addr = ret_dmi_addr;
      
      ret = DMI_write(dmi_scheduler.sthreads_addr + sthread_id * sizeof(sthread_t), sizeof(sthread_t), &sthread, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_mutex_unlock(dmi_scheduler.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    case STATE_JOIN:
      sthread.state = STATE_EXIT;
      sthread.dmi_addr = ret_dmi_addr;
      
      ret = DMI_write(dmi_scheduler.sthreads_addr + sthread_id * sizeof(sthread_t), sizeof(sthread_t), &sthread, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_cond_broadcast(dmi_scheduler.dmi_cond_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
            
      ret = DMI_mutex_unlock(dmi_scheduler.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    case STATE_EXIT:
      ret = DMI_mutex_unlock(dmi_scheduler.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      return FALSE;
      break;
    case STATE_DETACH:
      ret = DMI_read(dmi_scheduler.resources_addr + sthread.dmi_thread.dmi_id * sizeof(resource_t), sizeof(resource_t), &resource, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      resource.sthread_num--;
      
      ret = DMI_write(dmi_scheduler.resources_addr + sthread.dmi_thread.dmi_id * sizeof(resource_t), sizeof(resource_t), &resource, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_mutex_unlock(dmi_scheduler.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_idpool_put(dmi_scheduler.dmi_idpool_addr, sthread_id);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    default:
      error();
    }
  return TRUE;
}

void sub_sort_nodes(dmi_node_t *nodes, int32_t node_num)
{
  dmi_node_t node;
  int32_t i, j;
  
  for(i = 0; i < node_num; i++)
    {
      for(j = node_num - 1; j > i; j--)
        {
          if(strcmp(nodes[i].hostname, nodes[j].hostname) > 0)
            {
              node = nodes[i];
              nodes[i] = nodes[j];
              nodes[j] = node;
            }
        }
    }
  return;
}

void dmi_scheduler_monitor(int64_t dmi_scheduler_addr)
{
  dmi_scheduler_t dmi_scheduler;
  resource_t *resources;
  sthread_t *sthreads;
  migthread_t *migthreads;
  dmi_node_t dmi_node;
  dmi_node_t *dmi_nodes, *target_dmi_nodes;
  char str[FNAME_SIZE], line[FNAME_SIZE];
  int8_t init_flag, final_flag;
  int8_t *phases;
  int32_t ret, dmi_node_id, dmi_node_num, dmi_id, dmi_rank, dmi_pnum
    , sthread_id, sthread_rank, sthread_pnum, remain, i, count;
  
  ret = DMI_read(dmi_scheduler_addr, sizeof(dmi_scheduler_t), &dmi_scheduler, DMI_GET, NULL);
  catch(ret);
  
  resources = (resource_t*)my_malloc(sizeof(resource_t) * CONST_NODE_MAX);
  sthreads = (sthread_t*)my_malloc(sizeof(sthread_t) * CONST_STHREAD_MAX);
  phases = (int8_t*)my_malloc(sizeof(int8_t) * CONST_STHREAD_MAX);
  dmi_nodes = (dmi_node_t*)my_malloc(sizeof(dmi_node_t) * CONST_NODE_MAX);
  target_dmi_nodes = (dmi_node_t*)my_malloc(sizeof(dmi_node_t) * CONST_NODE_MAX);
  migthreads = (migthread_t*)my_malloc(sizeof(migthread_t) * CONST_STHREAD_MAX);
  
  init_flag = FALSE;
  while(1)
    {
      fprintf(stdout, "DMI scheduler >> ");
      fgets(line, FNAME_SIZE, stdin);
      sscanf(line, " %s\n", str);
      
      if(!strcmp(str, "schedule_perfect") || !strcmp(str, "p"))
        {
          ret = DMI_nodes(dmi_nodes, &dmi_node_num, CONST_NODE_MAX);
          catch(ret);
          
          for(dmi_node_id = 0; dmi_node_id < dmi_node_num; dmi_node_id++)
            {
              if(dmi_nodes[dmi_node_id].state == STATE_OPEN)
                {
                  ret = DMI_welcome(dmi_nodes[dmi_node_id].dmi_id);
                  catch(ret);
                }
            }
          
          ret = DMI_mutex_lock(dmi_scheduler.dmi_mutex_addr);
          catch(ret);
          
          ret = DMI_read(dmi_scheduler.resources_addr, sizeof(resource_t) * CONST_NODE_MAX, resources, DMI_GET, NULL);
          catch(ret);
          
          for(dmi_node_id = 0; dmi_node_id < dmi_node_num; dmi_node_id++)
            {
              dmi_node = dmi_nodes[dmi_node_id];
              if(dmi_node.state == STATE_OPEN)
                {
                  if(resources[dmi_node.dmi_id].sthread_num != 0) error();
                  if(resources[dmi_node.dmi_id].state != STATE_CLOSE) error();
                  resources[dmi_node.dmi_id].state = STATE_OPEN;
                  resources[dmi_node.dmi_id].dmi_node = dmi_node;
                }
              else if(dmi_node.state == STATE_CLOSE)
                {
                  resources[dmi_node.dmi_id].state = STATE_CLOSE;
                  resources[dmi_node.dmi_id].dmi_node = dmi_node;
                }
            }
          
          dmi_rank = 0;
          for(dmi_id = 0; dmi_id < CONST_NODE_MAX; dmi_id++)
            {
              if(resources[dmi_id].state == STATE_OPEN)
                {
                  for(i = 0; i < dmi_rank; i++)
                    {
                      if(!strcmp(target_dmi_nodes[i].hostname, resources[dmi_id].dmi_node.hostname))
                        {
                          break;
                        }
                    }
                  if(i == dmi_rank)
                    {
                      target_dmi_nodes[dmi_rank] = resources[dmi_id].dmi_node;
                      dmi_rank++;
                    }
                }
            }
          dmi_pnum = dmi_rank;
          
          if(init_flag == FALSE && dmi_pnum > 0)
            {
              ret = DMI_write(dmi_scheduler.resources_addr, sizeof(resource_t) * CONST_NODE_MAX, resources, DMI_EXCLUSIVE, NULL);
              catch(ret);
              
              init_flag = TRUE;
              ret = DMI_write(dmi_scheduler.init_flag_addr, sizeof(int8_t), &init_flag, DMI_PUT, NULL);
              catch(ret);
              
              ret = DMI_cond_broadcast(dmi_scheduler.dmi_cond_addr);
              catch(ret);
              
              ret = DMI_mutex_unlock(dmi_scheduler.dmi_mutex_addr);
              catch(ret);
            }
          else if(dmi_pnum > 0)
            {
              ret = DMI_read(dmi_scheduler.sthreads_addr, sizeof(sthread_t) * CONST_STHREAD_MAX, sthreads, DMI_GET, NULL);
              catch(ret);
              
              ret = DMI_mutex_lock(dmi_scheduler.dmi_mutex2_addr);
              catch(ret);
              
              ret = DMI_read(dmi_scheduler.phases_addr, sizeof(int8_t) * CONST_STHREAD_MAX, phases, DMI_GET, NULL);
              catch(ret);
              
              sthread_rank = 0;
              for(sthread_id = 0; sthread_id < CONST_STHREAD_MAX; sthread_id++)
                {
                  if((sthreads[sthread_id].state == STATE_CREATE
                      || sthreads[sthread_id].state == STATE_JOIN
                      || sthreads[sthread_id].state == STATE_DETACH)
                     && phases[sthread_id] == 0)
                    {
                      migthreads[sthread_rank].exit_flag = FALSE;
                      migthreads[sthread_rank].sthread_id = sthread_id;
                      for(dmi_id = 0; dmi_id < CONST_NODE_MAX; dmi_id++)
                        {
                          migthreads[sthread_rank].check_flags[dmi_id] = FALSE;
                        }
                      sthread_rank++;
                    }
                }
              sthread_pnum = sthread_rank;
              
              ret = DMI_mutex_unlock(dmi_scheduler.dmi_mutex2_addr);
              catch(ret);
              
              time_lap(1);
              
              dmi_rank = -1;
              remain = 0;
              for(sthread_rank = 0; sthread_rank < sthread_pnum; sthread_rank++)
                {
                  if(remain == 0)
                    {
                      dmi_rank++;
                      if(dmi_rank < sthread_pnum % dmi_pnum)
                        {
                          remain = sthread_pnum / dmi_pnum + 1;
                        }
                      else
                        {
                          remain = sthread_pnum / dmi_pnum;
                        }
                    }
                  remain--;
                  
                  migthreads[sthread_rank].target_dmi_node = target_dmi_nodes[dmi_rank];
                }
              if(!((dmi_rank == -1 && remain == 0) || (dmi_rank == dmi_pnum - 1 && remain == 0)))
                {
                  error();
                }
              
              count = 0;
              while(1)
                {
                  for(sthread_rank = 0; sthread_rank < sthread_pnum; sthread_rank++)
                    {
                      if(migthreads[sthread_rank].exit_flag == FALSE)
                        {
                          migthreads[sthread_rank].src_dmi_id = sthreads[migthreads[sthread_rank].sthread_id].dmi_thread.dmi_id;
                          migthreads[sthread_rank].dst_dmi_id = SYS_ID_UNDEF;
                          for(dmi_id = 0; dmi_id < CONST_NODE_MAX; dmi_id++)
                            {
                              if(migthreads[sthread_rank].check_flags[dmi_id] == FALSE
                                 && resources[dmi_id].state == STATE_OPEN
                                 && !strcmp(resources[dmi_id].dmi_node.hostname, migthreads[sthread_rank].target_dmi_node.hostname))
                                {
                                  migthreads[sthread_rank].dst_dmi_id = dmi_id;
                                  migthreads[sthread_rank].check_flags[dmi_id] = TRUE;
                                }
                            }
                          if(migthreads[sthread_rank].dst_dmi_id == SYS_ID_UNDEF)
                            {
                              ret = DMI_fork("", migthreads[sthread_rank].target_dmi_node.dmi_id, &dmi_node);
                              catch(ret);
                              
                              if(resources[dmi_node.dmi_id].sthread_num != 0) error();
                              if(resources[dmi_node.dmi_id].state != STATE_CLOSE) error();
                              resources[dmi_node.dmi_id].state = STATE_OPEN;
                              resources[dmi_node.dmi_id].dmi_node = dmi_node;
                              
                              migthreads[sthread_rank].dst_dmi_id = dmi_node.dmi_id;
                              migthreads[sthread_rank].check_flags[dmi_node.dmi_id] = TRUE;
                            }
                          
                          if(migthreads[sthread_rank].src_dmi_id != migthreads[sthread_rank].dst_dmi_id)
                            {
                              fprintf(stdout, "migrating thread %d (%s -> %s)\n"
                                      , migthreads[sthread_rank].sthread_id
                                      , resources[migthreads[sthread_rank].src_dmi_id].dmi_node.hostname
                                      , resources[migthreads[sthread_rank].dst_dmi_id].dmi_node.hostname);
                              
                              migthreads[sthread_rank].migrate_flag = FALSE;
                              
                              ret = DMI_migrate(sthreads[migthreads[sthread_rank].sthread_id].dmi_thread
                                                , migthreads[sthread_rank].dst_dmi_id
                                                , &sthreads[migthreads[sthread_rank].sthread_id].dmi_thread
                                                , &migthreads[sthread_rank].migrate_flag
                                                , &migthreads[sthread_rank].status);
                              catch(ret);
                            }
                          else
                            {
                              fprintf(stdout, "not migrating thread %d (%s)\n"
                                      , migthreads[sthread_rank].sthread_id
                                      , resources[migthreads[sthread_rank].src_dmi_id].dmi_node.hostname);
                              
                              migthreads[sthread_rank].exit_flag = TRUE;
                              count++;
                            }
                        }
                    }
                  
                  for(sthread_rank = 0; sthread_rank < sthread_pnum; sthread_rank++)
                    {
                      if(migthreads[sthread_rank].exit_flag == FALSE)
                        {
                          DMI_wait(&migthreads[sthread_rank].status, &ret);
                          if(migthreads[sthread_rank].migrate_flag == TRUE)
                            {
                              resources[migthreads[sthread_rank].src_dmi_id].sthread_num--;
                              resources[migthreads[sthread_rank].dst_dmi_id].sthread_num++;
                              
                              migthreads[sthread_rank].exit_flag = TRUE;
                              count++;
                            }
                        }
                    }
                  if(count == sthread_pnum)
                    {
                      break;
                    }
                }
              
              fprintf(stdout, "migration time = %.12lf sec\n", time_diff(1));
              
              ret = DMI_write(dmi_scheduler.sthreads_addr, sizeof(sthread_t) * CONST_STHREAD_MAX, sthreads, DMI_EXCLUSIVE, NULL);
              catch(ret);
              ret = DMI_write(dmi_scheduler.resources_addr, sizeof(resource_t) * CONST_NODE_MAX, resources, DMI_EXCLUSIVE, NULL);
              catch(ret);
              
              ret = DMI_mutex_unlock(dmi_scheduler.dmi_mutex_addr);
              catch(ret);
              
              for(dmi_node_id = 0; dmi_node_id < dmi_node_num; dmi_node_id++)
                {
                  if(dmi_nodes[dmi_node_id].state == STATE_CLOSE)
                    {
                      ret = DMI_goodbye(dmi_nodes[dmi_node_id].dmi_id);
                      catch(ret);
                    }
                }
            }
          else
            {
              ret = DMI_write(dmi_scheduler.resources_addr, sizeof(resource_t) * CONST_NODE_MAX, resources, DMI_EXCLUSIVE, NULL);
              catch(ret);
              
              ret = DMI_mutex_unlock(dmi_scheduler.dmi_mutex_addr);
              catch(ret);
              
              ret = DMI_read(dmi_scheduler.final_flag_addr, sizeof(int8_t), &final_flag, DMI_GET, NULL);
              catch(ret);
              if(final_flag == TRUE)
                {
                  for(dmi_node_id = 0; dmi_node_id < dmi_node_num; dmi_node_id++)
                    {
                      if(dmi_nodes[dmi_node_id].state == STATE_CLOSE)
                        {
                          ret = DMI_goodbye(dmi_nodes[dmi_node_id].dmi_id);
                          catch(ret);
                        }
                    }
                  break;
                }
            }
        }
      else
        {
          fprintf(stderr, "No such command (print 'help' to see all commands)\n");
        }
    }
      
  my_free(migthreads);
  my_free(sthreads);
  my_free(phases);
  my_free(dmi_nodes);
  my_free(target_dmi_nodes);
  my_free(resources);
  return;
}

/* malloc/realloc/free */

void* my_malloc_hook(int64_t size)
{
  void *p;
  
  if(_tls_thread_malloc_flag == TRUE)
    {
      _tls_thread_malloc_flag = FALSE;
      
      p = DMI_thread_malloc(size);
      
      _tls_thread_malloc_flag = TRUE;
    }
  else
    {
      p = malloc(size);
      //p = calloc(size, 1);
    }
  if(size != 0 && p == NULL) error();
  return p;
}

void* my_realloc_hook(void *old_p, int64_t size)
{
  void *p;
  
  if(_tls_thread_malloc_flag == TRUE)
    {
      _tls_thread_malloc_flag = FALSE;
      
      p = DMI_thread_realloc(old_p, size);
      
      _tls_thread_malloc_flag = TRUE;
    }
  else
    {
      p = realloc(old_p, size);
    }
  if(size != 0 && p == NULL) error();
  return p;
}

void my_free_hook(void *p)
{
  if(_tls_thread_malloc_flag == TRUE)
    {
      _tls_thread_malloc_flag = FALSE;
      
      DMI_thread_free(p);
      
      _tls_thread_malloc_flag = TRUE;
    }
  else
    {
      free(p);
    }
  return;
}

void* dmi_kr_malloc(int64_t size)
{
  wdmi_thread_t *wdmi_thread;
  mheader_t *mheader, *prev_mheader;
  void *ptr;
  int64_t mmap_size, malloc_size;
  static int64_t expand_size = CONST_THREAD_MALLOC_EXPAND_SIZE;
  
  if(size == 0)
    {
      return NULL;
    }
  
  wdmi_thread = _tls_wdmi_thread;
  if(wdmi_thread == NULL) error();
  
  malloc_size = (size + sizeof(mheader_t) + sizeof(mheader_t) - 1) / sizeof(mheader_t) * sizeof(mheader_t);
  if(malloc_size >= CONST_THREAD_MALLOC_MMAP_SIZE)
    {
      mmap_size = SYS_PAGESIZE_ALIGN_CEIL(malloc_size);
      
      mheader = (mheader_t*)DMI_thread_mmap(NULL, mmap_size, PROT_READ | PROT_WRITE, MAP_PRIVATE | MAP_ANONYMOUS, -1, 0);
      if(mheader == MAP_FAILED)
        {
          ptr = NULL;
        }
      else
        {
          mheader->size = -mmap_size;
          ptr = mheader + 1;
        }
    }
  else
    {
      prev_mheader = wdmi_thread->head_mheader;
      mheader = prev_mheader->next_mheader;
      while(1)
        {
          if(mheader->size >= malloc_size)
            {
              if(mheader->size == malloc_size)
                {
                  prev_mheader->next_mheader = mheader->next_mheader;
                }
              else
                {
                  mheader->size -= malloc_size;
                  mheader += mheader->size / sizeof(mheader_t);
                  mheader->size = malloc_size;
                }
              wdmi_thread->head_mheader = prev_mheader;
              ptr = mheader + 1;
              break;
            }
          if(mheader == wdmi_thread->head_mheader)
            {
              if(malloc_size < expand_size)
                {
                  mmap_size = expand_size;
                  //expand_size *= 2;
                }
              else
                {
                  mmap_size = SYS_PAGESIZE_ALIGN_CEIL(malloc_size);
                }
              
              mheader = (mheader_t*)DMI_thread_mmap(NULL, mmap_size, PROT_READ | PROT_WRITE, MAP_PRIVATE | MAP_ANONYMOUS, -1, 0);
              
              if(mheader == MAP_FAILED)
                {
                  ptr = NULL;
                  break;
                }
              mheader->size = mmap_size;
              DMI_thread_free(mheader + 1);
              mheader = wdmi_thread->head_mheader;
            }
          prev_mheader = mheader;
          mheader = mheader->next_mheader;
        }
    }
  
  if(ptr == NULL) error();
  
  /*memset(ptr, 0, size);*/
  return ptr;
}

void* dmi_kr_realloc(void *ptr, int64_t size)
{
  mheader_t *mheader, *realloc_mheader;
  void *new_ptr;
  int64_t mmap_size, malloc_size, copy_size, old_size;
  
  if(ptr == NULL)
    {
      new_ptr = DMI_thread_malloc(size);
    }
  else if(size == 0)
    {
      DMI_thread_free(ptr);
      new_ptr = NULL;
    }
  else
    {
      new_ptr = NULL;
      realloc_mheader = (mheader_t*)ptr - 1;
      if(realloc_mheader->size < 0)
        {
          malloc_size = (size + sizeof(mheader_t) + sizeof(mheader_t) - 1) / sizeof(mheader_t) * sizeof(mheader_t);
          mmap_size = SYS_PAGESIZE_ALIGN_CEIL(malloc_size);
          
          mheader = (mheader_t*)DMI_thread_mremap(realloc_mheader, -realloc_mheader->size, mmap_size, MREMAP_MAYMOVE);
          if(mheader == MAP_FAILED)
            {
              new_ptr = NULL;
            }
          else
            {
              mheader->size = -mmap_size;
              new_ptr = mheader + 1;
            }
        }
      
      if(new_ptr == NULL)
        {
          new_ptr = DMI_thread_malloc(size);
          mheader = (mheader_t*)new_ptr - 1;
          if(new_ptr != NULL)
            {
              old_size = realloc_mheader->size - sizeof(mheader_t);
              copy_size = old_size > size ? size : old_size;
              memcpy(new_ptr, ptr, copy_size);
              
              DMI_thread_free(ptr);
            }
        }
    }
  if(new_ptr == NULL) error();
  return new_ptr;
}

void dmi_kr_free(void *ptr)
{
  wdmi_thread_t *wdmi_thread;
  mheader_t *free_mheader, *mheader;
  int32_t ret;
  
  if(ptr == NULL)
    {
      return;
    }
  
  wdmi_thread = _tls_wdmi_thread;
  if(wdmi_thread == NULL) error();
  
  free_mheader = (mheader_t*)ptr - 1;
  if(free_mheader->size < 0)
    {
      ret = DMI_thread_munmap(free_mheader, -free_mheader->size);
      if(ret < 0) error();
    }
  else
    {
      for(mheader = wdmi_thread->head_mheader; !(mheader < free_mheader && free_mheader < mheader->next_mheader); mheader = mheader->next_mheader)
        {
          if(mheader >= mheader->next_mheader && (mheader < free_mheader || free_mheader < mheader->next_mheader))
            {
              break;
            }
        }
      if(free_mheader + free_mheader->size / sizeof(mheader_t) == mheader->next_mheader)
        {
          free_mheader->size += mheader->next_mheader->size;
          free_mheader->next_mheader = mheader->next_mheader->next_mheader;
        }
      else
        {
          free_mheader->next_mheader = mheader->next_mheader;
        }
      if(mheader + mheader->size / sizeof(mheader_t) == free_mheader)
        {
          mheader->size += free_mheader->size;
          mheader->next_mheader = free_mheader->next_mheader;
        }
      else
        {
          mheader->next_mheader = free_mheader;
        }
      wdmi_thread->head_mheader = mheader;
    }
  return;
}

/* mutex/cond */

int32_t dmi_spin_init(int64_t dmi_spinlock_addr)
{
  int8_t flag;
  int32_t ret;
  
  flag = FALSE;
  ret = DMI_write((int64_t)&(((dmi_spinlock_t*)dmi_spinlock_addr)->flag), sizeof(int8_t), &flag, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_spin_destroy(int64_t dmi_spinlock_addr)
{
  return TRUE;
}

int32_t dmi_spin_lock(int64_t dmi_spinlock_addr)
{
  int8_t cmp_flag, swap_flag, cas_flag;
  int32_t ret;
  
  do
    {
      cmp_flag = FALSE;
      swap_flag = TRUE;
      ret = DMI_cas((int64_t)&(((dmi_spinlock_t*)dmi_spinlock_addr)->flag), sizeof(int8_t), &cmp_flag, &swap_flag, &cas_flag, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  while(cas_flag == FALSE);
  return TRUE;
}

int32_t dmi_spin_unlock(int64_t dmi_spinlock_addr)
{
  int8_t flag;
  int32_t ret;
  
  flag = FALSE;
  ret = DMI_write((int64_t)&(((dmi_spinlock_t*)dmi_spinlock_addr)->flag), sizeof(int8_t), &flag, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_spin_trylock(int64_t dmi_spinlock_addr, int32_t *try_flag_ptr)
{
  int8_t cmp_flag, swap_flag, cas_flag;
  int32_t ret;
  
  cmp_flag = FALSE;
  swap_flag = TRUE;
  ret = DMI_cas((int64_t)&(((dmi_spinlock_t*)dmi_spinlock_addr)->flag), sizeof(int8_t), &cmp_flag, &swap_flag, &cas_flag, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  *try_flag_ptr = cas_flag;
  return TRUE;
}

int32_t dmi_mutex_init(int64_t dmi_mutex_addr)
{
  dmi_mutex_t dmi_mutex;
  int32_t ret;
  
  dmi_mutex.head_addr = DMI_NULL;
  dmi_mutex.next_addr = DMI_NULL;
  dmi_mutex.stopper1_addr = DMI_NULL;
  dmi_mutex.stopper2_addr = DMI_NULL;
  
  ret = DMI_write(dmi_mutex_addr, sizeof(dmi_mutex_t), &dmi_mutex, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_mutex_destroy(int64_t dmi_mutex_addr)
{
  return TRUE;
}

int32_t dmi_mutex_lock(int64_t dmi_mutex_addr)
{
  dmi_mutex_t dmi_mutex;
  wdmi_thread_t *wdmi_thread;
  int32_t ret;
  int64_t prev_addr, curr_addr;
  
  wdmi_thread = _tls_wdmi_thread;
  if(wdmi_thread == NULL) error();
  
  ret = DMI_read(dmi_mutex_addr, sizeof(dmi_mutex_t), &dmi_mutex, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  curr_addr = sub_convert_addr(wdmi_thread->stackarea_addr, dmi_mutex.stopper1_addr, dmi_mutex.stopper2_addr);
  
#if DEBUG_MUTEX
  ret = DMI_fas((int64_t)&(((dmi_mutex_t*)dmi_mutex_addr)->head_addr), sizeof(int64_t), &curr_addr, &prev_addr, DMI_PUT, NULL);
#else
  ret = DMI_fas((int64_t)&(((dmi_mutex_t*)dmi_mutex_addr)->head_addr), sizeof(int64_t), &curr_addr, &prev_addr, DMI_PUT, NULL);
#endif
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  if(prev_addr == DMI_NULL)
    {
#if DEBUG_MUTEX
      ret = DMI_write((int64_t)&(((dmi_mutex_t*)dmi_mutex_addr)->stopper1_addr), sizeof(int64_t), &curr_addr, DMI_PUT, NULL);
#else
      ret = DMI_write((int64_t)&(((dmi_mutex_t*)dmi_mutex_addr)->stopper1_addr), sizeof(int64_t), &curr_addr, DMI_PUT, NULL);
#endif
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  else
    {
      ret = DMI_suspend(NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
#if DEBUG_MUTEX
  ret = DMI_write((int64_t)&(((dmi_mutex_t*)dmi_mutex_addr)->next_addr), sizeof(int64_t), &prev_addr, DMI_PUT, NULL);
#else
  ret = DMI_write((int64_t)&(((dmi_mutex_t*)dmi_mutex_addr)->next_addr), sizeof(int64_t), &prev_addr, DMI_PUT, NULL);
#endif
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_mutex_unlock(int64_t dmi_mutex_addr)
{
  dmi_mutex_t dmi_mutex;
  dmi_thread_t dmi_thread;
  int8_t cas_flag;
  int32_t ret;
  int64_t dmi_addr, stackarea_addr;
  
  ret = DMI_read(dmi_mutex_addr, sizeof(dmi_mutex_t), &dmi_mutex, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  if(dmi_mutex.next_addr == DMI_NULL || dmi_mutex.next_addr == dmi_mutex.stopper1_addr)
    {
      if(dmi_mutex.next_addr == dmi_mutex.stopper1_addr)
        {
          dmi_mutex.stopper1_addr = dmi_mutex.stopper2_addr;
#if DEBUG_MUTEX
          ret = DMI_write((int64_t)&(((dmi_mutex_t*)dmi_mutex_addr)->stopper1_addr), sizeof(int64_t), &dmi_mutex.stopper1_addr, DMI_PUT, NULL);
#else
          ret = DMI_write((int64_t)&(((dmi_mutex_t*)dmi_mutex_addr)->stopper1_addr), sizeof(int64_t), &dmi_mutex.stopper1_addr, DMI_PUT, NULL);
#endif
          throw_or_catch(_wdmi->config->catch_flag, ret);
        }
     
      dmi_addr = DMI_NULL;
#if DEBUG_MUTEX 
      ret = DMI_cas((int64_t)&(((dmi_mutex_t*)dmi_mutex_addr)->head_addr), sizeof(int64_t), &dmi_mutex.stopper1_addr, &dmi_addr, &cas_flag, DMI_PUT, NULL);
#else
      ret = DMI_cas((int64_t)&(((dmi_mutex_t*)dmi_mutex_addr)->head_addr), sizeof(int64_t), &dmi_mutex.stopper1_addr, &dmi_addr, &cas_flag, DMI_PUT, NULL);
#endif
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      if(cas_flag == FALSE)
        {
          ret = DMI_read(dmi_mutex_addr, sizeof(dmi_mutex_t), &dmi_mutex, DMI_GET, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
          
#if DEBUG_MUTEX
          ret = DMI_write((int64_t)&(((dmi_mutex_t*)dmi_mutex_addr)->stopper2_addr), sizeof(int64_t), &dmi_mutex.head_addr, DMI_PUT, NULL);
#else
          ret = DMI_write((int64_t)&(((dmi_mutex_t*)dmi_mutex_addr)->stopper2_addr), sizeof(int64_t), &dmi_mutex.head_addr, DMI_PUT, NULL);
#endif
          throw_or_catch(_wdmi->config->catch_flag, ret);
          
          stackarea_addr = sub_revert_addr(dmi_mutex.head_addr);
          
          ret = DMI_read((int64_t)&(((stackarea_t*)stackarea_addr)->dmi_thread), sizeof(dmi_thread_t), &dmi_thread, DMI_GET, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
          
          ret = DMI_wake(dmi_thread, NULL, 0, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
        }
    }
  else
    {
      stackarea_addr = sub_revert_addr(dmi_mutex.next_addr);
      
      ret = DMI_read((int64_t)&(((stackarea_t*)stackarea_addr)->dmi_thread), sizeof(dmi_thread_t), &dmi_thread, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_wake(dmi_thread, NULL, 0, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t dmi_mutex_trylock(int64_t dmi_mutex_addr, int32_t *try_flag_ptr)
{
  dmi_mutex_t dmi_mutex;
  wdmi_thread_t *wdmi_thread;
  int8_t cas_flag;
  int32_t ret;
  int64_t dmi_addr, curr_addr;
  
  wdmi_thread = _tls_wdmi_thread;
  if(wdmi_thread == NULL) error();
  
  ret = DMI_read(dmi_mutex_addr, sizeof(dmi_mutex_t), &dmi_mutex, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  curr_addr = sub_convert_addr(wdmi_thread->stackarea_addr, dmi_mutex.stopper1_addr, dmi_mutex.stopper2_addr);
  dmi_addr = DMI_NULL;
  
  ret = DMI_cas((int64_t)&(((dmi_mutex_t*)dmi_mutex_addr)->head_addr), sizeof(int64_t), &dmi_addr, &curr_addr, &cas_flag, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  if(cas_flag == TRUE)
    {
      ret = DMI_write((int64_t)&(((dmi_mutex_t*)dmi_mutex_addr)->stopper1_addr), sizeof(int64_t), &curr_addr, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      dmi_addr = DMI_NULL;
      ret = DMI_write((int64_t)&(((dmi_mutex_t*)dmi_mutex_addr)->next_addr), sizeof(int64_t), &dmi_addr, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      *try_flag_ptr = TRUE;
    }
  else
    {
      *try_flag_ptr = FALSE;
    }
  return TRUE;
}

int32_t dmi_cond_init(int64_t dmi_cond_addr)
{
  dmi_cond_t dmi_cond;
  int32_t ret;
  
  dmi_cond.head1_addr = DMI_NULL;
  dmi_cond.head2_addr = DMI_NULL;
  dmi_cond.state = STATE_NONE;
  
  ret = DMI_write(dmi_cond_addr, sizeof(dmi_cond_t), &dmi_cond, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_cond_destroy(int64_t dmi_cond_addr)
{
  return TRUE;
}

int32_t dmi_cond_wait(int64_t dmi_cond_addr, int64_t dmi_mutex_addr)
{
  dmi_cond_t dmi_cond;
  wdmi_thread_t *wdmi_thread;
  int32_t ret;
  
  wdmi_thread = _tls_wdmi_thread;
  if(wdmi_thread == NULL) error();
  
  ret = DMI_read(dmi_cond_addr, sizeof(dmi_cond_t), &dmi_cond, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  if(dmi_cond.state == STATE_NONE)
    {
      ret = DMI_write((int64_t)&(((stackarea_t*)wdmi_thread->stackarea_addr)->next_addr), sizeof(int64_t), &dmi_cond.head1_addr, DMI_EXCLUSIVE, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      dmi_cond.state = STATE_HEAD1;
      dmi_cond.head1_addr = wdmi_thread->stackarea_addr;
      ret = DMI_write(dmi_cond_addr, sizeof(dmi_cond_t), &dmi_cond, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  else if(dmi_cond.state == STATE_HEAD1)
    {
      ret = DMI_write((int64_t)&(((stackarea_t*)wdmi_thread->stackarea_addr)->next_addr), sizeof(int64_t), &dmi_cond.head1_addr, DMI_EXCLUSIVE, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      dmi_cond.head1_addr = wdmi_thread->stackarea_addr;
      ret = DMI_write(dmi_cond_addr, sizeof(dmi_cond_t), &dmi_cond, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_read(dmi_cond_addr, sizeof(dmi_cond_t), &dmi_cond, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  else if(dmi_cond.state == STATE_HEAD2)
    {
      ret = DMI_write((int64_t)&(((stackarea_t*)wdmi_thread->stackarea_addr)->next_addr), sizeof(int64_t), &dmi_cond.head2_addr, DMI_EXCLUSIVE, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      dmi_cond.head2_addr = wdmi_thread->stackarea_addr;
      ret = DMI_write(dmi_cond_addr, sizeof(dmi_cond_t), &dmi_cond, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  else
    {
      error();
    }
  
  ret = DMI_mutex_unlock(dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_suspend(NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_lock(dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_cond_signal(int64_t dmi_cond_addr)
{
  dmi_cond_t dmi_cond;
  stackarea_t stackarea;
  int32_t ret;
  
  ret = DMI_read(dmi_cond_addr, sizeof(dmi_cond_t), &dmi_cond, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  if(dmi_cond.state == STATE_NONE)
    {
    }
  else if(dmi_cond.state == STATE_HEAD1)
    {
      if(dmi_cond.head2_addr != DMI_NULL)
        {
          ret = DMI_read(dmi_cond.head2_addr, sizeof(stackarea_t), &stackarea, DMI_GET, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
          
          dmi_cond.head2_addr = stackarea.next_addr;
          ret = DMI_write(dmi_cond_addr, sizeof(dmi_cond_t), &dmi_cond, DMI_PUT, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
          
          ret = DMI_wake(stackarea.dmi_thread, NULL, 0, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
        }
      else
        {
          if(dmi_cond.head1_addr != DMI_NULL)
            {
              ret = DMI_read(dmi_cond.head1_addr, sizeof(stackarea_t), &stackarea, DMI_GET, NULL);
              throw_or_catch(_wdmi->config->catch_flag, ret);
              
              dmi_cond.state = STATE_HEAD2;
              dmi_cond.head1_addr = stackarea.next_addr;
              ret = DMI_write(dmi_cond_addr, sizeof(dmi_cond_t), &dmi_cond, DMI_PUT, NULL);
              throw_or_catch(_wdmi->config->catch_flag, ret);
              
              ret = DMI_wake(stackarea.dmi_thread, NULL, 0, NULL);
              throw_or_catch(_wdmi->config->catch_flag, ret);
            }
          else
            {
              dmi_cond.state = STATE_NONE;
              ret = DMI_write(dmi_cond_addr, sizeof(dmi_cond_t), &dmi_cond, DMI_PUT, NULL);
              throw_or_catch(_wdmi->config->catch_flag, ret);
            }
        }
    }
  else if(dmi_cond.state == STATE_HEAD2)
    {
      if(dmi_cond.head1_addr != DMI_NULL)
        {
          ret = DMI_read(dmi_cond.head1_addr, sizeof(stackarea_t), &stackarea, DMI_GET, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
          
          dmi_cond.head1_addr = stackarea.next_addr;
          ret = DMI_write(dmi_cond_addr, sizeof(dmi_cond_t), &dmi_cond, DMI_PUT, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
          
          ret = DMI_wake(stackarea.dmi_thread, NULL, 0, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
        }
      else
        {
          if(dmi_cond.head2_addr != DMI_NULL)
            {
              ret = DMI_read(dmi_cond.head2_addr, sizeof(stackarea_t), &stackarea, DMI_GET, NULL);
              throw_or_catch(_wdmi->config->catch_flag, ret);
              
              dmi_cond.state = STATE_HEAD1;
              dmi_cond.head2_addr = stackarea.next_addr;
              ret = DMI_write(dmi_cond_addr, sizeof(dmi_cond_t), &dmi_cond, DMI_PUT, NULL);
              throw_or_catch(_wdmi->config->catch_flag, ret);
              
              ret = DMI_wake(stackarea.dmi_thread, NULL, 0, NULL);
              throw_or_catch(_wdmi->config->catch_flag, ret);
            }
          else
            {
              dmi_cond.state = STATE_NONE;
              ret = DMI_write(dmi_cond_addr, sizeof(dmi_cond_t), &dmi_cond, DMI_PUT, NULL);
              throw_or_catch(_wdmi->config->catch_flag, ret);
            }
        }
    }
  else
    {
      error();
    }
  return TRUE;
}

int32_t dmi_cond_broadcast(int64_t dmi_cond_addr)
{
  dmi_cond_t dmi_cond;
  int64_t next_addr;
  stackarea_t stackarea;
  int32_t ret;
  
  ret = DMI_read(dmi_cond_addr, sizeof(dmi_cond_t), &dmi_cond, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  next_addr = dmi_cond.head1_addr;
  while(next_addr != DMI_NULL)
    {
      ret = DMI_read(next_addr, sizeof(stackarea_t), &stackarea, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_wake(stackarea.dmi_thread, NULL, 0, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      next_addr = stackarea.next_addr;
    }
  
  next_addr = dmi_cond.head2_addr;
  while(next_addr != DMI_NULL)
    {
      ret = DMI_read(next_addr, sizeof(stackarea_t), &stackarea, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_wake(stackarea.dmi_thread, NULL, 0, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      next_addr = stackarea.next_addr;
    }
  
  dmi_cond.head1_addr = DMI_NULL;
  dmi_cond.head2_addr = DMI_NULL;
  dmi_cond.state = STATE_NONE;
  ret = DMI_write(dmi_cond_addr, sizeof(dmi_cond_t), &dmi_cond, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  return TRUE;
}

int32_t dmi_idpool_init(int64_t dmi_idpool_addr, int32_t id_max)
{
  dmi_idpool_t dmi_idpool;
  int32_t id, ret, sp;
  
  dmi_idpool.id_max = id_max;
  ret = DMI_mmap(&dmi_idpool.unused_addr, sizeof(int32_t) * dmi_idpool.id_max, 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&dmi_idpool.sp_addr, sizeof(int32_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&dmi_idpool.dmi_spinlock_addr, sizeof(dmi_spinlock_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  for(id = 0; id < dmi_idpool.id_max; id++)
    {
      ret = DMI_write(dmi_idpool.unused_addr + id * sizeof(int32_t), sizeof(int32_t), &id, DMI_EXCLUSIVE, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  sp = 0;
  ret = DMI_write(dmi_idpool.sp_addr, sizeof(int32_t), &sp, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_spin_init(dmi_idpool.dmi_spinlock_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_write(dmi_idpool_addr, sizeof(dmi_idpool_t), &dmi_idpool, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_idpool_destroy(int64_t dmi_idpool_addr)
{
  dmi_idpool_t dmi_idpool;
  int32_t ret;
  
  ret = DMI_read(dmi_idpool_addr, sizeof(dmi_idpool_t), &dmi_idpool, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_spin_destroy(dmi_idpool.dmi_spinlock_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_munmap(dmi_idpool.dmi_spinlock_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_idpool.sp_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_idpool.unused_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_idpool_get(int64_t dmi_idpool_addr, int32_t *id_ptr)
{
  dmi_idpool_t dmi_idpool;
  int32_t ret, sp;
  
  ret = DMI_read(dmi_idpool_addr, sizeof(dmi_idpool_t), &dmi_idpool, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_spin_lock(dmi_idpool.dmi_spinlock_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_read(dmi_idpool.sp_addr, sizeof(int32_t), &sp, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_read(dmi_idpool.unused_addr + sp * sizeof(int32_t), sizeof(int32_t), id_ptr, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  sp++;
  if(sp >= dmi_idpool.id_max)
    {
      error();
    }
  
  ret = DMI_write(dmi_idpool.sp_addr, sizeof(int32_t), &sp, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_spin_unlock(dmi_idpool.dmi_spinlock_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_idpool_put(int64_t dmi_idpool_addr, int32_t id)
{
  dmi_idpool_t dmi_idpool;
  int32_t ret, sp;
  
  ret = DMI_read(dmi_idpool_addr, sizeof(dmi_idpool_t), &dmi_idpool, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_spin_lock(dmi_idpool.dmi_spinlock_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_read(dmi_idpool.sp_addr, sizeof(int32_t), &sp, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  --sp;
  if(sp >= dmi_idpool.id_max)
    {
      error();
    }
  
  ret = DMI_write(dmi_idpool.unused_addr + sp * sizeof(int32_t), sizeof(int32_t), &id, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_write(dmi_idpool.sp_addr, sizeof(int32_t), &sp, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_spin_unlock(dmi_idpool.dmi_spinlock_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

/* rwset */

int32_t dmi_rwset_init(int64_t dmi_rwset_addr, int64_t element_num, int64_t element_size, int32_t rwset_num)
{
  dmi_rwset_t dmi_rwset;
  domain_t *domains;
  int32_t ret, id;
  int64_t page_num, page_size;
  
  dmi_rwset.rwset_num = rwset_num;
  dmi_rwset.element_num = element_num;
  dmi_rwset.element_size = element_size;
  
  page_num = rwset_num * 20;
  page_size = sizeof(int64_t) * dmi_rwset.element_num / page_num;
  page_num += 1;
  if(page_size < sizeof(int64_t))
    {
      page_size = sizeof(int64_t) * dmi_rwset.element_num;
      page_num = 1;
    }
#if DEBUG_MMAP
#if DEBUG_BASIC
  fprintf(stderr, "correct page size (dmi_rwset.write_addrs_addr) : %ld\n", sizeof(int64_t) * dmi_rwset.element_num);
#endif
  ret = DMI_mmap(&dmi_rwset.write_addrs_addr, DMI_AUTOMATIC, sizeof(int64_t) * dmi_rwset.element_num, NULL);
#else
  ret = DMI_mmap(&dmi_rwset.write_addrs_addr, page_size, page_num, NULL);
#endif
  throw_or_catch(_wdmi->config->catch_flag, ret);
  _dmi_memory += sizeof(int64_t) * dmi_rwset.element_num;

#if DEBUG_MMAP
#if DEBUG_BASIC
  fprintf(stderr, "correct page size (dmi_rwset.domains_addr) : %ld\n", sizeof(domain_t) * dmi_rwset.rwset_num);
#endif
  ret = DMI_mmap(&dmi_rwset.domains_addr, DMI_AUTOMATIC, sizeof(domain_t) * dmi_rwset.rwset_num, NULL);
#else
  ret = DMI_mmap(&dmi_rwset.domains_addr, sizeof(domain_t) * dmi_rwset.rwset_num, 1, NULL);
#endif
  throw_or_catch(_wdmi->config->catch_flag, ret);
  _dmi_memory += sizeof(domain_t) * dmi_rwset.rwset_num;
  
  domains = (domain_t*)my_malloc(dmi_rwset.rwset_num * sizeof(domain_t));
  for(id = 0; id < dmi_rwset.rwset_num; id++)
    {
      domains[id].write_addr = DMI_NULL;
      domains[id].write_element_num = 0;
    }
  ret = DMI_write(dmi_rwset.domains_addr, sizeof(domain_t) * dmi_rwset.rwset_num, domains, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  my_free(domains);
  
  ret = DMI_write(dmi_rwset_addr, sizeof(dmi_rwset_t), &dmi_rwset, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_rwset_destroy(int64_t dmi_rwset_addr)
{
  dmi_rwset_t dmi_rwset;
  domain_t *domains;
  int32_t ret, id;
  
  ret = DMI_read(dmi_rwset_addr, sizeof(dmi_rwset_t), &dmi_rwset, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  domains = (domain_t*)my_malloc(sizeof(domain_t) * dmi_rwset.rwset_num);
  
  ret = DMI_read(dmi_rwset.domains_addr, sizeof(domain_t) * dmi_rwset.rwset_num, domains, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  for(id = 0; id < dmi_rwset.rwset_num; id++)
    {
      if(domains[id].write_addr != DMI_NULL)
        {
          ret = DMI_munmap(domains[id].write_addr, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
        }
    }
  
  my_free(domains);
  
  ret = DMI_munmap(dmi_rwset.domains_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_rwset.write_addrs_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_rwset_decompose(int64_t dmi_rwset_addr, int32_t my_id, int64_t *write_elements, int32_t write_element_num)
{
  dmi_rwset_t dmi_rwset;
  domain_t domain;
  group_t write_addrs_group;
  int64_t *write_addrs;
  int32_t ret;
  int64_t seq, write_addr;
  int64_t *addrs, *ptr_offsets, *write_sizes;
  
  ret = DMI_read(dmi_rwset_addr, sizeof(dmi_rwset_t), &dmi_rwset, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  if(!(0 <= my_id && my_id < dmi_rwset.rwset_num)) return FALSE;
  
#if DEBUG_MMAP
#if DEBUG_BASIC
  fprintf(stderr, "correct page size (dmi_rwset.write_addr) : %ld\n", write_element_num * dmi_rwset.element_size);
#endif
  ret = DMI_mmap(&write_addr, DMI_AUTOMATIC, write_element_num * dmi_rwset.element_size, NULL);
#else
  ret = DMI_mmap(&write_addr, write_element_num * dmi_rwset.element_size, 1, NULL);
#endif
  throw_or_catch(_wdmi->config->catch_flag, ret);
  _dmi_memory += write_element_num * dmi_rwset.element_size;
  
  write_addrs = (int64_t*)my_malloc(sizeof(int64_t) * write_element_num);
  addrs = (int64_t*)my_malloc(sizeof(int64_t) * write_element_num);
  ptr_offsets = (int64_t*)my_malloc(sizeof(int64_t) * write_element_num);
  write_sizes = (int64_t*)my_malloc(sizeof(int64_t) * write_element_num);
  
  for(seq = 0; seq < write_element_num; seq++)
    {
      write_addrs[seq] = write_addr + seq * dmi_rwset.element_size;
      addrs[seq] = dmi_rwset.write_addrs_addr + write_elements[seq] * sizeof(int64_t);
      ptr_offsets[seq] = seq * sizeof(int64_t);
      write_sizes[seq] = sizeof(int64_t);
    }
  
  ret = DMI_group_init(&write_addrs_group, addrs, ptr_offsets, write_sizes, write_element_num);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_group_write(&write_addrs_group, write_addrs, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_group_destroy(&write_addrs_group);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  my_free(write_sizes);
  my_free(addrs);
  my_free(ptr_offsets);
  my_free(write_addrs);
  
  domain.write_addr = write_addr;
  domain.write_element_num = write_element_num;
  ret = DMI_write(dmi_rwset.domains_addr + my_id * sizeof(domain_t), sizeof(domain_t), &domain, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

rwset_t* rwset_alloc(void)
{
  rwset_t *rwset;
  
  rwset = (rwset_t*)my_malloc(sizeof(rwset_t));
  return rwset;
}

void rwset_free(rwset_t *rwset)
{
  my_free(rwset);
  return;
}

int32_t rwset_init(rwset_t *rwset, int64_t dmi_rwset_addr, int32_t my_id, int64_t *read_elements, int32_t read_element_num)
{
  dmi_rwset_t dmi_rwset;
  domain_t domain;
  group_t read_addrs_group;
  int32_t ret;
  int64_t seq, old_seq, old_write_addr, read_num;
  int64_t *read_addrs, *addrs, *ptr_offsets, *read_sizes;
  
  ret = DMI_read(dmi_rwset_addr, sizeof(dmi_rwset_t), &dmi_rwset, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  if(!(0 <= my_id && my_id < dmi_rwset.rwset_num)) return FALSE;
  
  ret = DMI_read(dmi_rwset.domains_addr + my_id * sizeof(domain_t), sizeof(domain_t), &domain, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  rwset->write_addr = domain.write_addr;
  rwset->write_size = domain.write_element_num * dmi_rwset.element_size;
  
  read_addrs = (int64_t*)my_malloc(sizeof(int64_t) * read_element_num);
  addrs = (int64_t*)my_malloc(sizeof(int64_t) * read_element_num);
  ptr_offsets = (int64_t*)my_malloc(sizeof(int64_t) * read_element_num);
  read_sizes = (int64_t*)my_malloc(sizeof(int64_t) * read_element_num);
  
  for(seq = 0; seq < read_element_num; seq++)
    {
      addrs[seq] = dmi_rwset.write_addrs_addr + read_elements[seq] * sizeof(int64_t);
      ptr_offsets[seq] = seq * sizeof(int64_t);
      read_sizes[seq] = sizeof(int64_t);
    }
  
  ret = DMI_group_init(&read_addrs_group, addrs, ptr_offsets, read_sizes, read_element_num);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_group_read(&read_addrs_group, read_addrs, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_group_destroy(&read_addrs_group);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  read_num = 0;
  if(read_element_num > 0)
    {
      seq = 0;
      old_seq = seq;
      old_write_addr = read_addrs[seq];
      for(seq = 1; seq < read_element_num; seq++)
        {
          if(read_addrs[seq] == DMI_NULL)
            {
              my_free(read_addrs);
              my_free(addrs);
              my_free(ptr_offsets);
              my_free(read_sizes);
              return FALSE;
            }
          if(read_addrs[seq] - old_write_addr != dmi_rwset.element_size)
            {
              addrs[read_num] = read_addrs[old_seq];
              ptr_offsets[read_num] = old_seq * dmi_rwset.element_size;
              read_sizes[read_num] = (seq - old_seq) * dmi_rwset.element_size;
              old_seq = seq;
              read_num++;
            }
          old_write_addr = read_addrs[seq];
        }
      addrs[read_num] = read_addrs[old_seq];
      ptr_offsets[read_num] = old_seq * dmi_rwset.element_size;
      read_sizes[read_num] = (seq - old_seq) * dmi_rwset.element_size;
      read_num++;
    }
  
  ret = DMI_group_init(&rwset->read_group, addrs, ptr_offsets, read_sizes, read_num);
  throw_or_catch(_wdmi->config->catch_flag, ret);
    
  my_free(read_addrs);
  my_free(addrs);
  my_free(ptr_offsets);
  my_free(read_sizes);
  return TRUE;
}

int32_t rwset_destroy(rwset_t *rwset)
{
  int32_t ret;
  
  ret = DMI_group_destroy(&rwset->read_group);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  return TRUE;
}

int32_t rwset_write(rwset_t *rwset, void *buf, status_t *status)
{
  int32_t ret;
  
  ret = DMI_write(rwset->write_addr, rwset->write_size, buf, DMI_EXCLUSIVE, status);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t rwset_read(rwset_t *rwset, void *buf, status_t *status)
{
  int32_t ret;
  
  ret = DMI_group_read(&rwset->read_group, buf, DMI_GET, status);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_rwset2_init(int64_t dmi_rwset2_addr, int64_t element_num, int64_t element_size, int32_t rwset2_num)
{
  dmi_rwset2_t dmi_rwset2;
  int32_t ret;
  
  dmi_rwset2.rwset2_num = rwset2_num;
  dmi_rwset2.element_size = element_size;
  dmi_rwset2.element_num = element_num;
  
#if DEBUG_MMAP
#if DEBUG_BASIC
  fprintf(stderr, "correct page size (dmi_rwset2.idids_addr) : %ld\n", sizeof(idid_t) * dmi_rwset2.element_num);
#endif
  ret = DMI_mmap(&dmi_rwset2.idids_addr, DMI_AUTOMATIC, sizeof(idid_t) * dmi_rwset2.element_num, NULL);
#else
  ret = DMI_mmap(&dmi_rwset2.idids_addr, sizeof(idid_t) * dmi_rwset2.element_num, 1, NULL);
#endif
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
#if DEBUG_MMAP
#if DEBUG_BASIC
  fprintf(stderr, "correct page size (dmi_rwset2.elements_addrs_addr) : %ld\n", sizeof(int64_t) * dmi_rwset2.rwset2_num);
#endif
  ret = DMI_mmap(&dmi_rwset2.read_elements_addrs_addr, DMI_AUTOMATIC, sizeof(int64_t) * dmi_rwset2.rwset2_num, NULL);
#else
  ret = DMI_mmap(&dmi_rwset2.read_elements_addrs_addr, sizeof(int64_t) * dmi_rwset2.rwset2_num, 1, NULL);
#endif
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
#if DEBUG_MMAP
#if DEBUG_BASIC
  fprintf(stderr, "correct page size (dmi_rwset2.read_element_nums_addr) : %ld\n", sizeof(int64_t) * dmi_rwset2.rwset2_num);
#endif
  ret = DMI_mmap(&dmi_rwset2.read_element_nums_addr, DMI_AUTOMATIC, sizeof(int64_t) * dmi_rwset2.rwset2_num, NULL);
#else
  ret = DMI_mmap(&dmi_rwset2.read_element_nums_addr, sizeof(int64_t) * dmi_rwset2.rwset2_num, 1, NULL);
#endif
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
#if DEBUG_MMAP
#if DEBUG_BASIC
  fprintf(stderr, "correct page size (dmi_rwset2.read_addrs_addr) : %ld\n", sizeof(int64_t) * dmi_rwset2.rwset2_num);
#endif
  ret = DMI_mmap(&dmi_rwset2.read_addrs_addr, DMI_AUTOMATIC, sizeof(int64_t) * dmi_rwset2.rwset2_num, NULL);
#else
  ret = DMI_mmap(&dmi_rwset2.read_addrs_addr, sizeof(int64_t) * dmi_rwset2.rwset2_num, 1, NULL);
#endif
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_write(dmi_rwset2_addr, sizeof(dmi_rwset2_t), &dmi_rwset2, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_rwset2_destroy(int64_t dmi_rwset2_addr)
{
  dmi_rwset2_t dmi_rwset2;
  int32_t ret, id;
  int64_t *addrs;
  
  ret = DMI_read(dmi_rwset2_addr, sizeof(dmi_rwset2_t), &dmi_rwset2, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  addrs = (int64_t*)my_malloc(sizeof(int64_t) * dmi_rwset2.rwset2_num);
  
  ret = DMI_read(dmi_rwset2.read_addrs_addr, sizeof(int64_t) * dmi_rwset2.rwset2_num, addrs, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  for(id = 0; id < dmi_rwset2.rwset2_num; id++)
    {
      ret = DMI_munmap(addrs[id], NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  ret = DMI_read(dmi_rwset2.read_elements_addrs_addr, sizeof(int64_t) * dmi_rwset2.rwset2_num, addrs, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  for(id = 0; id < dmi_rwset2.rwset2_num; id++)
    {
      ret = DMI_munmap(addrs[id], NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  my_free(addrs);
  
  ret = DMI_munmap(dmi_rwset2.read_addrs_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_rwset2.read_element_nums_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_rwset2.read_elements_addrs_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_rwset2.idids_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_rwset2_decompose(int64_t dmi_rwset2_addr, int32_t my_id, int64_t *write_elements, int32_t write_element_num, int64_t *read_elements, int32_t read_element_num)
{
  dmi_rwset2_t dmi_rwset2;
  idid_t *idids;
  group_t idids_group;
  int32_t ret;
  int64_t seq, read_addr, read_elements_addr;
  int64_t *addrs, *ptr_offsets, *sizes;
  
  ret = DMI_read(dmi_rwset2_addr, sizeof(dmi_rwset2_t), &dmi_rwset2, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  if(!(0 <= my_id && my_id < dmi_rwset2.rwset2_num)) return FALSE;
  
  idids = (idid_t*)my_malloc(sizeof(idid_t) * write_element_num);
  addrs = (int64_t*)my_malloc(sizeof(int64_t) * write_element_num);
  ptr_offsets = (int64_t*)my_malloc(sizeof(int64_t) * write_element_num);
  sizes = (int64_t*)my_malloc(sizeof(int64_t) * write_element_num);
  
  for(seq = 0; seq < write_element_num; seq++)
    {
      idids[seq].id1 = my_id;
      idids[seq].id2 = seq;
      addrs[seq] = dmi_rwset2.idids_addr + write_elements[seq] * sizeof(idid_t);
      ptr_offsets[seq] = seq * sizeof(idid_t);
      sizes[seq] = sizeof(idid_t);
    }
  
  ret = DMI_group_init(&idids_group, addrs, ptr_offsets, sizes, write_element_num);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_group_write(&idids_group, idids, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_group_destroy(&idids_group);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  my_free(addrs);
  my_free(ptr_offsets);
  my_free(sizes);
  my_free(idids);
  
#if DEBUG_MMAP
#if DEBUG_BASIC
  fprintf(stderr, "correct page size (dmi_rwset2.read_addr) : %ld\n", dmi_rwset2.element_size * read_element_num);
#endif
  ret = DMI_mmap(&read_addr, DMI_AUTOMATIC, dmi_rwset2.element_size * read_element_num, NULL);
#else
  ret = DMI_mmap(&read_addr, dmi_rwset2.element_size * read_element_num, 1, NULL);
#endif
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_write(dmi_rwset2.read_addrs_addr + my_id * sizeof(int64_t), sizeof(int64_t), &read_addr, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
#if DEBUG_MMAP
#if DEBUG_BASIC
  fprintf(stderr, "correct page size (dmi_rwset2.read_elements_addr) : %ld\n", sizeof(int64_t) * read_element_num);
#endif
  ret = DMI_mmap(&read_elements_addr, DMI_AUTOMATIC, sizeof(int64_t) * read_element_num, NULL);
#else
  ret = DMI_mmap(&read_elements_addr, sizeof(int64_t) * read_element_num, 1, NULL);
#endif
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_write(read_elements_addr, sizeof(int64_t) * read_element_num, read_elements, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_write(dmi_rwset2.read_elements_addrs_addr + my_id * sizeof(int64_t), sizeof(int64_t), &read_elements_addr, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_write(dmi_rwset2.read_element_nums_addr + my_id * sizeof(int32_t), sizeof(int32_t), &read_element_num, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

rwset2_t* rwset2_alloc(void)
{
  rwset2_t *rwset2;
  
  rwset2 = (rwset2_t*)my_malloc(sizeof(rwset2_t));
  return rwset2;
}

void rwset2_free(rwset2_t *rwset2)
{
  my_free(rwset2);
  return;
}

int32_t rwset2_init(rwset2_t *rwset2, int64_t dmi_rwset2_addr, int32_t my_id)
{
  dmi_rwset2_t dmi_rwset2;
  idid_t *idids;
  int32_t ret, read_element_num, group_num, group_capacity, id;
  int64_t seq, element, read_elements_addr;
  int64_t *read_addrs, *read_elements, *ptr_offsets, *addrs, *write_sizes;
  
  ret = DMI_read(dmi_rwset2_addr, sizeof(dmi_rwset2_t), &dmi_rwset2, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  if(!(0 <= my_id && my_id < dmi_rwset2.rwset2_num)) return FALSE;
  
  read_addrs = (int64_t*)my_malloc(sizeof(int64_t) * dmi_rwset2.rwset2_num);
  
  ret = DMI_read(dmi_rwset2.read_addrs_addr, sizeof(int64_t) * dmi_rwset2.rwset2_num, read_addrs, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  rwset2->read_addr = read_addrs[my_id];
  
  ret = DMI_read(dmi_rwset2.read_element_nums_addr + my_id * sizeof(int32_t), sizeof(int32_t), &read_element_num, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  rwset2->read_size = read_element_num * dmi_rwset2.element_size;
  
  idids = (idid_t*)my_malloc(sizeof(idid_t) * dmi_rwset2.element_num);
  
  ret = DMI_read(dmi_rwset2.idids_addr, sizeof(idid_t) * dmi_rwset2.element_num, idids, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  group_capacity = read_element_num * 2;
  addrs = (int64_t*)my_malloc(sizeof(int64_t) * group_capacity);
  ptr_offsets = (int64_t*)my_malloc(sizeof(int64_t) * group_capacity);
  write_sizes = (int64_t*)my_malloc(sizeof(int64_t) * group_capacity);
  
  group_num = 0;
  for(id = 0; id < dmi_rwset2.rwset2_num; id++)
    {
      ret = DMI_read(dmi_rwset2.read_elements_addrs_addr + id * sizeof(int64_t), sizeof(int64_t), &read_elements_addr, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_read(dmi_rwset2.read_element_nums_addr + id * sizeof(int32_t), sizeof(int32_t), &read_element_num, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      read_elements = (int64_t*)my_malloc(sizeof(int64_t) * read_element_num);
      
      ret = DMI_read(read_elements_addr, sizeof(int64_t) * read_element_num, read_elements, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      for(seq = 0; seq < read_element_num; seq++)
        {
          element = read_elements[seq];
          if(idids[element].id1 == my_id)
            {
              if(group_num >= group_capacity)
                {
                  group_capacity *= 2;
                  
                  addrs = (int64_t*)my_realloc(addrs, sizeof(int64_t) * group_capacity);
                  ptr_offsets = (int64_t*)my_realloc(ptr_offsets, sizeof(int64_t) * group_capacity);
                  write_sizes = (int64_t*)my_realloc(write_sizes, sizeof(int64_t) * group_capacity);
                }
              
              addrs[group_num] = read_addrs[id] + seq * dmi_rwset2.element_size;
              ptr_offsets[group_num] = idids[element].id2 * dmi_rwset2.element_size;
              write_sizes[group_num] = dmi_rwset2.element_size;
              group_num++;
            }
        }
      
      my_free(read_elements);
    }
  
  rwset2->addrs = (int64_t*)DMI_thread_malloc(sizeof(int64_t) * group_num);
  rwset2->ptr_offsets = (int64_t*)DMI_thread_malloc(sizeof(int64_t) * group_num);
  rwset2->write_sizes = (int64_t*)DMI_thread_malloc(sizeof(int64_t) * group_num);
  
  memcpy(rwset2->addrs, addrs, sizeof(int64_t) * group_num);
  memcpy(rwset2->ptr_offsets, ptr_offsets, sizeof(int64_t) * group_num);
  memcpy(rwset2->write_sizes, write_sizes, sizeof(int64_t) * group_num);
  
  ret = DMI_group_init(&rwset2->write_group, rwset2->addrs, rwset2->ptr_offsets, rwset2->write_sizes, group_num);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  my_free(addrs);
  my_free(ptr_offsets);
  my_free(write_sizes);
  
  my_free(idids);
  my_free(read_addrs);
  return TRUE;
}

int32_t rwset2_destroy(rwset2_t *rwset2)
{
  int32_t ret;
  
  ret = DMI_group_destroy(&rwset2->write_group);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  DMI_thread_free(rwset2->addrs);
  DMI_thread_free(rwset2->ptr_offsets);
  DMI_thread_free(rwset2->write_sizes);
  return TRUE;
}

int32_t rwset2_write(rwset2_t *rwset2, void *buf, status_t *status)
{
  int32_t ret;
  
  ret = DMI_group_write(&rwset2->write_group, buf, DMI_PUT, status);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t rwset2_read(rwset2_t *rwset2, void *buf, status_t *status)
{
  int32_t ret;
  
  ret = DMI_write(rwset2->read_addr, 0, NULL, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_read(rwset2->read_addr, rwset2->read_size, buf, DMI_INVALIDATE, status);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

/* barrier */

int32_t dmi_barrier_init(int64_t dmi_barrier_addr)
{
  dmi_barrier_t dmi_barrier;
  int ret;
  int64_t i_values[3];
  double d_values[3];
  
#if DEBUG_MMAP
#if DEBUG_BASIC
  fprintf(stderr, "correct page size (dmi_barrier.double_sum_addr) : %ld\n", sizeof(double) * 3);
#endif
  ret = DMI_mmap(&dmi_barrier.double_sum_addr, DMI_AUTOMATIC, sizeof(double) * 3, NULL);
#else
  ret = DMI_mmap(&dmi_barrier.double_sum_addr, sizeof(double) * 3, 1, NULL);
#endif
  throw_or_catch(_wdmi->config->catch_flag, ret);
#if DEBUG_MMAP
#if DEBUG_BASIC
  fprintf(stderr, "correct page size (dmi_barrier.double_center_addr) : %ld\n", sizeof(double) * 3);
#endif
  ret = DMI_mmap(&dmi_barrier.double_center_addr, DMI_AUTOMATIC, sizeof(double) * 3, NULL);
#else
  ret = DMI_mmap(&dmi_barrier.double_center_addr, sizeof(double) * 3, 1, NULL);
#endif
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  d_values[0] = SYS_DOUBLE_DUMMY;
  d_values[1] = SYS_DOUBLE_DUMMY;
  d_values[2] = SYS_DOUBLE_DUMMY;
  ret = DMI_write(dmi_barrier.double_sum_addr, sizeof(double) * 3, d_values, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  d_values[0] = 0;
  d_values[1] = 0;
  d_values[2] = 0;
  ret = DMI_write(dmi_barrier.double_center_addr, sizeof(double) * 3, d_values, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
#if DEBUG_MMAP
#if DEBUG_BASIC
  fprintf(stderr, "correct page size (dmi_barrier.int64_sum_addr) : %ld\n", sizeof(int64_t) * 3);
#endif
  ret = DMI_mmap(&dmi_barrier.int64_sum_addr, DMI_AUTOMATIC, sizeof(int64_t) * 3, NULL);
#else
  ret = DMI_mmap(&dmi_barrier.int64_sum_addr, sizeof(int64_t) * 3, 1, NULL);
#endif
  throw_or_catch(_wdmi->config->catch_flag, ret);
#if DEBUG_MMAP
#if DEBUG_BASIC
  fprintf(stderr, "correct page size (dmi_barrier.int64_center_addr) : %ld\n", sizeof(int64_t) * 3);
#endif
  ret = DMI_mmap(&dmi_barrier.int64_center_addr, DMI_AUTOMATIC, sizeof(int64_t) * 3, NULL);
#else
  ret = DMI_mmap(&dmi_barrier.int64_center_addr, sizeof(int64_t) * 3, 1, NULL);
#endif
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  i_values[0] = SYS_INT64_DUMMY;
  i_values[1] = SYS_INT64_DUMMY;
  i_values[2] = SYS_INT64_DUMMY;
  ret = DMI_write(dmi_barrier.int64_sum_addr, sizeof(int64_t) * 3, i_values, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  i_values[0] = 0;
  i_values[1] = 0;
  i_values[2] = 0;
  ret = DMI_write(dmi_barrier.int64_center_addr, sizeof(int64_t) * 3, i_values, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_write(dmi_barrier_addr, sizeof(dmi_barrier_t), &dmi_barrier, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_barrier_destroy(int64_t dmi_barrier_addr)
{
  dmi_barrier_t dmi_barrier;
  int ret;
  
  ret = DMI_read(dmi_barrier_addr, sizeof(dmi_barrier_t), &dmi_barrier, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_munmap(dmi_barrier.int64_center_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_barrier.double_center_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_barrier.int64_sum_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_barrier.double_sum_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

barrier_t* barrier_alloc(void)
{
  barrier_t *barrier;
  
  barrier = (barrier_t*)my_malloc(sizeof(barrier_t));
  return barrier;
}

void barrier_free(barrier_t *barrier)
{
  my_free(barrier);
  return;
}

int32_t barrier_init(barrier_t *barrier, int64_t dmi_barrier_addr)
{
  dmi_barrier_t dmi_barrier;
  int32_t ret;
  
  ret = DMI_read(dmi_barrier_addr, sizeof(dmi_barrier_t), &dmi_barrier, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  barrier->int64_sum_addr = dmi_barrier.int64_sum_addr;
  barrier->int64_center_addr = dmi_barrier.int64_center_addr;
  barrier->double_sum_addr = dmi_barrier.double_sum_addr;
  barrier->double_center_addr = dmi_barrier.double_center_addr;
  return TRUE;
}

int32_t barrier_destroy(barrier_t *barrier)
{
  return TRUE;
}

int32_t barrier_allreduce_int64(barrier_t *barrier, int32_t sync_num, int64_t sub_sum, int64_t *sum_ptr, int8_t op_type)
{
  int ret;
  int64_t dummy;
  int64_t in_buf[2], i_values[3], out_buf[3];
  
  if(!(op_type == DMI_OP_MAX || op_type == DMI_OP_MIN ||
       op_type == DMI_OP_SUM || op_type == DMI_OP_PROD))
    {
      return FALSE;
    }
  
  out_buf[0] = sub_sum;
  out_buf[1] = sync_num;
  out_buf[2] = op_type;
  
  ret = DMI_atomic(barrier->int64_center_addr, sizeof(int64_t) * 3, out_buf, sizeof(int64_t) * 3, in_buf, sizeof(int64_t) * 2, SYS_INT64_BARRIER_TAG, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  if(in_buf[0] != SYS_INT64_DUMMY)
    {
      if(in_buf[1] == 0)
        {
          i_values[0] = in_buf[0];
          i_values[1] = SYS_INT64_DUMMY;
        }
      else if(in_buf[1] == 1)
        {
          i_values[1] = in_buf[0];
          i_values[2] = SYS_INT64_DUMMY;
        }
      else
        {
          i_values[2] = in_buf[0];
          i_values[0] = SYS_INT64_DUMMY;
        }
      
      ret = DMI_write(barrier->int64_sum_addr, sizeof(int64_t) * 3, i_values, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  else
    {
      dummy = SYS_INT64_DUMMY;
      ret = DMI_watch(barrier->int64_sum_addr + sizeof(int64_t) * in_buf[1], sizeof(int64_t), &in_buf[0], &dummy, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  if(sum_ptr != NULL)
    {
      *sum_ptr = in_buf[0];
    }
  return TRUE;
}

int32_t barrier_allreduce_double(barrier_t *barrier, int32_t sync_num, double sub_sum, double *sum_ptr, int8_t op_type)
{
  int ret;
  double dummy;
  double in_buf[2], d_values[3], out_buf[3];
  
  if(!(op_type == DMI_OP_MAX || op_type == DMI_OP_MIN ||
       op_type == DMI_OP_SUM || op_type == DMI_OP_PROD))
    {
      return FALSE;
    }
  
  out_buf[0] = sub_sum;
  out_buf[1] = (double)sync_num;
  out_buf[2] = (double)op_type;
  
  ret = DMI_atomic(barrier->double_center_addr, sizeof(double) * 3, out_buf, sizeof(double) * 3, in_buf, sizeof(double) * 2, SYS_DOUBLE_BARRIER_TAG, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  if(in_buf[0] != SYS_DOUBLE_DUMMY)
    {
      if((int)in_buf[1] == 0)
        {
          d_values[0] = in_buf[0];
          d_values[1] = SYS_DOUBLE_DUMMY;
        }
      else if((int)in_buf[1] == 1)
        {
          d_values[1] = in_buf[0];
          d_values[2] = SYS_DOUBLE_DUMMY;
        }
      else
        {
          d_values[2] = in_buf[0];
          d_values[0] = SYS_DOUBLE_DUMMY;
        }
      
      ret = DMI_write(barrier->double_sum_addr, sizeof(double) * 3, d_values, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  else
    {
#if 1
      dummy = SYS_DOUBLE_DUMMY;
      ret = DMI_watch(barrier->double_sum_addr + sizeof(double) * (int)in_buf[1], sizeof(double), &in_buf[0], &dummy, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
#else
      dummy = SYS_DOUBLE_DUMMY;
      do
        {
          ret = DMI_read(barrier->double_sum_addr + sizeof(double) * (int)in_buf[1], sizeof(double), &in_buf[0], DMI_GET, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
        }
      while(in_buf[0] == dummy);
#endif
    }
  
  if(sum_ptr != NULL)
    {
      *sum_ptr = in_buf[0];
    }
  return TRUE;
}

/* mthread */

int32_t dmi_mthread_init(int64_t dmi_mthread_addr)
{
  dmi_mthread_t dmi_mthread;
  int32_t ret;
  
  ret = DMI_mmap(&dmi_mthread.stackareaes_addr, sizeof(stackarea_t), CONST_MTHREAD_MAX, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&dmi_mthread.dmi_idpool_addr, sizeof(dmi_idpool_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_idpool_init(dmi_mthread.dmi_idpool_addr, CONST_MTHREAD_MAX);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_write(dmi_mthread_addr, sizeof(dmi_mthread_t), &dmi_mthread, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_mthread_destroy(int64_t dmi_mthread_addr)
{
  dmi_mthread_t dmi_mthread;
  int32_t ret;
  
  ret = DMI_read(dmi_mthread_addr, sizeof(dmi_mthread_t), &dmi_mthread, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_idpool_destroy(dmi_mthread.dmi_idpool_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_munmap(dmi_mthread.dmi_idpool_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_mthread.stackareaes_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_mthread_get(int64_t dmi_mthread_addr, int32_t *mthread_id_ptr, int64_t *stackarea_addr_ptr)
{
  dmi_mthread_t dmi_mthread;
  int32_t ret, mthread_id;
  
  ret = DMI_read(dmi_mthread_addr, sizeof(dmi_mthread_t), &dmi_mthread, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_idpool_get(dmi_mthread.dmi_idpool_addr, &mthread_id);
  throw_or_catch(_wdmi->config->catch_flag, ret);
    
  *mthread_id_ptr = mthread_id;
  *stackarea_addr_ptr = dmi_mthread.stackareaes_addr + mthread_id * sizeof(stackarea_t);
  return TRUE;
}

int32_t dmi_mthread_put(int64_t dmi_mthread_addr, int32_t mthread_id)
{
  dmi_mthread_t dmi_mthread;
  int32_t ret;
  
  ret = DMI_read(dmi_mthread_addr, sizeof(dmi_mthread_t), &dmi_mthread, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_idpool_put(dmi_mthread.dmi_idpool_addr, mthread_id);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

/* member */

int32_t dmi_member_init(int64_t dmi_member_addr)
{
  dmi_member_t dmi_member;
  dmi_node_t dmi_node;
  int32_t dmi_id, ret, node_num, rescale_node_num;
  
  ret = DMI_read(dmi_member_addr, sizeof(dmi_member_t), &dmi_member, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mmap(&dmi_member.dmi_nodes_addr, sizeof(dmi_node_t), CONST_NODE_MAX, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  for(dmi_id = 0; dmi_id < CONST_NODE_MAX; dmi_id++)
    {
      dmi_node.state = STATE_CLOSED;
      dmi_node.dmi_id = -1;
      dmi_node.core = 0;
      dmi_node.memory = 0;
      
      ret = DMI_write(dmi_member.dmi_nodes_addr + dmi_id * sizeof(dmi_node_t), sizeof(dmi_node_t), &dmi_node, DMI_EXCLUSIVE, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  ret = DMI_mmap(&dmi_member.dmi_mutex_addr, sizeof(dmi_mutex_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&dmi_member.dmi_cond_addr, sizeof(dmi_cond_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&dmi_member.node_num_addr, sizeof(int64_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&dmi_member.rescale_node_num_addr, sizeof(int64_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_init(dmi_member.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_cond_init(dmi_member.dmi_cond_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  node_num = 0;
  ret = DMI_write(dmi_member.node_num_addr, sizeof(int32_t), &node_num, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  rescale_node_num = 0;
  ret = DMI_write(dmi_member.rescale_node_num_addr, sizeof(int32_t), &rescale_node_num, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_write(dmi_member_addr, sizeof(dmi_member_t), &dmi_member, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_member_destroy(int64_t dmi_member_addr)
{
  dmi_member_t dmi_member;
  int32_t ret;
  
  ret = DMI_read(dmi_member_addr, sizeof(dmi_member_t), &dmi_member, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_cond_destroy(dmi_member.dmi_cond_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_destroy(dmi_member.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_munmap(dmi_member.node_num_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_member.rescale_node_num_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_member.dmi_cond_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_member.dmi_mutex_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_member.dmi_nodes_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_member_join(int64_t dmi_member_addr, int32_t dmi_id, int32_t core_num, int64_t mem_size, dmi_node_t *dmi_node_ptr, int8_t detach_flag)
{
  dmi_member_t dmi_member;
  dmi_node_t dmi_node;
  int32_t ret, i;
  
  ret = DMI_read(dmi_member_addr, sizeof(dmi_member_t), &dmi_member, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_lock(dmi_member.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  if(detach_flag == FALSE)
    {
      dmi_node.state = STATE_OPEN;
    }
  else
    {
      dmi_node.state = STATE_OPENED;
    }
  
  dmi_node.dmi_id = dmi_id;
  dmi_node.core = core_num;
  dmi_node.memory = mem_size;
  
  ret = gethostname(dmi_node.hostname, IP_SIZE);
  if(ret < 0) error();
  for(i = strlen(dmi_node.hostname); i < IP_SIZE; i++)
    {
      dmi_node.hostname[i] = 0;
    }
  
  ret = DMI_write(dmi_member.dmi_nodes_addr + dmi_id * sizeof(dmi_node_t), sizeof(dmi_node_t), &dmi_node, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  if(detach_flag == FALSE)
    {
      ret = DMI_cond_broadcast(dmi_member.dmi_cond_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      while(1)
        {
          ret = DMI_read(dmi_member.dmi_nodes_addr + dmi_id * sizeof(dmi_node_t), sizeof(dmi_node_t), &dmi_node, DMI_INVALIDATE, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
          
          if(dmi_node.state == STATE_OPENED)
            {
              break;
            }
          
          ret = DMI_cond_wait(dmi_member.dmi_cond_addr, dmi_member.dmi_mutex_addr);
          throw_or_catch(_wdmi->config->catch_flag, ret);
        }
    }
  
  ret = DMI_mutex_unlock(dmi_member.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  *dmi_node_ptr = dmi_node;
  return TRUE;
}

int32_t dmi_member_leave(int64_t dmi_member_addr, int32_t dmi_id, int8_t detach_flag)
{
  dmi_member_t dmi_member;
  dmi_node_t dmi_node;
  int32_t ret, state;
  
  ret = DMI_read(dmi_member_addr, sizeof(dmi_member_t), &dmi_member, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_lock(dmi_member.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  if(detach_flag == FALSE)
    {
      state = STATE_CLOSE;
    }
  else
    {
      state = STATE_CLOSED;
    }
  
  ret = DMI_write(dmi_member.dmi_nodes_addr + dmi_id * sizeof(dmi_node_t), sizeof(int32_t), &state, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  if(detach_flag == FALSE)
    {  
      ret = DMI_cond_broadcast(dmi_member.dmi_cond_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      while(1)
        {
          ret = DMI_read(dmi_member.dmi_nodes_addr + dmi_id * sizeof(dmi_node_t), sizeof(dmi_node_t), &dmi_node, DMI_INVALIDATE, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
          
          if(dmi_node.state == STATE_CLOSED)
            {
              break;
            }
          
          ret = DMI_cond_wait(dmi_member.dmi_cond_addr, dmi_member.dmi_mutex_addr);
          throw_or_catch(_wdmi->config->catch_flag, ret);
        }
    }
  
  ret = DMI_mutex_unlock(dmi_member.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_member_welcome(int64_t dmi_member_addr, int32_t dmi_id)
{
  dmi_member_t dmi_member;
  dmi_node_t dmi_node;
  int32_t ret, state;
  
  ret = DMI_read(dmi_member_addr, sizeof(dmi_member_t), &dmi_member, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_lock(dmi_member.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  while(1)
    {
      ret = DMI_read(dmi_member.dmi_nodes_addr + dmi_id * sizeof(dmi_node_t), sizeof(dmi_node_t), &dmi_node, DMI_INVALIDATE, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      if(dmi_node.state == STATE_OPEN)
        {
          break;
        }
      
      ret = DMI_cond_wait(dmi_member.dmi_cond_addr, dmi_member.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  state = STATE_OPENED;
  ret = DMI_write(dmi_member.dmi_nodes_addr + dmi_id * sizeof(dmi_node_t), sizeof(int32_t), &state, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_cond_broadcast(dmi_member.dmi_cond_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_unlock(dmi_member.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_member_goodbye(int64_t dmi_member_addr, int32_t dmi_id)
{
  dmi_member_t dmi_member;
  dmi_node_t dmi_node;
  int32_t ret, state;
  
  ret = DMI_read(dmi_member_addr, sizeof(dmi_member_t), &dmi_member, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_lock(dmi_member.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  while(1)
    {
      ret = DMI_read(dmi_member.dmi_nodes_addr + dmi_id * sizeof(dmi_node_t), sizeof(dmi_node_t), &dmi_node, DMI_INVALIDATE, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      if(dmi_node.state == STATE_CLOSE)
        {
          break;
        }
      
      ret = DMI_cond_wait(dmi_member.dmi_cond_addr, dmi_member.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  state = STATE_CLOSED;
  ret = DMI_write(dmi_member.dmi_nodes_addr + dmi_id * sizeof(dmi_node_t), sizeof(int32_t), &state, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_cond_broadcast(dmi_member.dmi_cond_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_unlock(dmi_member.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_member_poll(int64_t dmi_member_addr, dmi_node_t *dmi_node_ptr)
{
  dmi_member_t dmi_member;
  dmi_node_t dmi_node;
  int8_t exit_flag;
  int32_t ret, dmi_id;
  
  ret = DMI_read(dmi_member_addr, sizeof(dmi_member_t), &dmi_member, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_lock(dmi_member.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  exit_flag = FALSE;
  while(1)
    {
      for(dmi_id = 0; dmi_id < CONST_NODE_MAX; dmi_id++)
        {
          ret = DMI_read(dmi_member.dmi_nodes_addr + dmi_id * sizeof(dmi_node_t), sizeof(dmi_node_t), &dmi_node, DMI_INVALIDATE, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
          
          if(dmi_node.state == STATE_OPEN || dmi_node.state == STATE_CLOSE)
            {
              *dmi_node_ptr = dmi_node;
              
              exit_flag = TRUE;
              break;
            }
        }
      if(exit_flag == TRUE)
        {
          break;
        }
      
      ret = DMI_cond_wait(dmi_member.dmi_cond_addr, dmi_member.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  ret = DMI_mutex_unlock(dmi_member.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_member_peek(int64_t dmi_member_addr, dmi_node_t *dmi_node_ptr, int8_t *flag_ptr)
{
  dmi_member_t dmi_member;
  dmi_node_t dmi_node;
  int32_t ret, dmi_id;
  
  ret = DMI_read(dmi_member_addr, sizeof(dmi_member_t), &dmi_member, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_lock(dmi_member.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  for(dmi_id = 0; dmi_id < CONST_NODE_MAX; dmi_id++)
    {
      ret = DMI_read(dmi_member.dmi_nodes_addr + dmi_id * sizeof(dmi_node_t), sizeof(dmi_node_t), &dmi_node, DMI_INVALIDATE, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      if(dmi_node.state == STATE_OPEN || dmi_node.state == STATE_CLOSE)
        {
          *dmi_node_ptr = dmi_node;
          break;
        }
    }
  if(dmi_id == CONST_NODE_MAX)
    {
      *flag_ptr = FALSE;
    }
  else
    {
      *flag_ptr = TRUE;
    }
  
  ret = DMI_mutex_unlock(dmi_member.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_member_nodes(int64_t dmi_member_addr, dmi_node_t *dmi_node_array, int32_t *num_ptr, int32_t capacity)
{
  dmi_member_t dmi_member;
  dmi_node_t *dmi_nodes;
  int32_t ret, count, dmi_id;
  
  ret = DMI_read(dmi_member_addr, sizeof(dmi_member_t), &dmi_member, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  dmi_nodes = (dmi_node_t*)my_malloc(sizeof(dmi_node_t) * CONST_NODE_MAX);
  
  ret = DMI_read(dmi_member.dmi_nodes_addr, sizeof(dmi_node_t) * CONST_NODE_MAX, dmi_nodes, DMI_INVALIDATE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  count = 0;
  for(dmi_id = 0; dmi_id < CONST_NODE_MAX && count < capacity; dmi_id++)
    {
      if(dmi_nodes[dmi_id].state != STATE_CLOSED)
        {
          dmi_node_array[count] = dmi_nodes[dmi_id];
          count++;
        }
    }
  
  my_free(dmi_nodes);
  
  *num_ptr = count;
  return TRUE;
}

int32_t dmi_member_fad(int64_t dmi_member_addr, int32_t add_value, int32_t *fetch_value_ptr)
{
  dmi_member_t dmi_member;
  int32_t ret;
  int64_t fetch_value;
  
  ret = DMI_read(dmi_member_addr, sizeof(dmi_member_t), &dmi_member, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_fad(dmi_member.node_num_addr, (int64_t)add_value, &fetch_value, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  if(fetch_value_ptr != NULL)
    {
      *fetch_value_ptr = (int32_t)fetch_value;
    }
  return TRUE;
}

int32_t dmi_member_rescalefad(int64_t dmi_member_addr, int32_t add_value, int32_t *fetch_value_ptr)
{
  dmi_member_t dmi_member;
  int32_t ret;
  int64_t fetch_value;
  
  ret = DMI_read(dmi_member_addr, sizeof(dmi_member_t), &dmi_member, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_fad(dmi_member.rescale_node_num_addr, (int32_t)add_value, &fetch_value, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  if(fetch_value_ptr != NULL)
    {
      *fetch_value_ptr = (int32_t)fetch_value;
    }
  return TRUE;
}

/* memalloc */

int32_t dmi_memalloc_init(int64_t dmi_memalloc_addr)
{
  dmi_memalloc_t dmi_memalloc;
  mheader2_t mheader2;
  int32_t ret;
  
  ret = DMI_mmap(&dmi_memalloc.dummy_mheader2_addr, sizeof(mheader2_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  mheader2.next_mheader2_addr = dmi_memalloc.dummy_mheader2_addr;
  mheader2.size = 0;
  ret = DMI_write(dmi_memalloc.dummy_mheader2_addr, sizeof(mheader2_t), &mheader2, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  dmi_memalloc.head_mheader2_addr_addr = dmi_memalloc.dummy_mheader2_addr;
  dmi_memalloc.dmi_mutex = DMI_MUTEX_INITIALIZER;
  
  ret = DMI_write(dmi_memalloc_addr, sizeof(dmi_memalloc_t), &dmi_memalloc, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_memalloc_destroy(int64_t dmi_memalloc_addr)
{
  dmi_memalloc_t dmi_memalloc;
  int32_t ret;
  
  ret = DMI_read(dmi_memalloc_addr, sizeof(dmi_memalloc_t), &dmi_memalloc, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_munmap(dmi_memalloc.dummy_mheader2_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_memalloc_malloc(int64_t dmi_memalloc_addr, int64_t *addr_ptr, int64_t size, int64_t page_size, int8_t rec_flag)
{
  mheader2_t mheader2, prev_mheader2;
  int32_t ret;
  int64_t mmap_size, malloc_size, prev_mheader2_addr, mheader2_addr, head_mheader2_addr, page_num;
  static int64_t expand_size = CONST_THREAD_MALLOC_EXPAND_SIZE;
  
  if(rec_flag == FALSE)
    {
      ret = DMI_mutex_lock((int64_t)&(((dmi_memalloc_t*)dmi_memalloc_addr)->dmi_mutex));
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  malloc_size = (size + sizeof(mheader2_t) + sizeof(mheader2_t) - 1) / sizeof(mheader2_t) * sizeof(mheader2_t);
  if(malloc_size >= CONST_DMI_MALLOC_MMAP_SIZE)
    {
      if(malloc_size < expand_size)
        {
          mmap_size = expand_size;
          //expand_size *= 2;
        }
      else
        {
          mmap_size = malloc_size;
        }
      if(page_size <= 0)
        {
          page_size = 4096;
        }
      page_num = mmap_size / page_size + 1;
      
      ret = DMI_mmap(&mheader2_addr, page_size, page_num, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      mheader2.size = -page_size * page_num;
      ret = DMI_write(mheader2_addr, sizeof(mheader2_t), &mheader2, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      *addr_ptr = mheader2_addr + sizeof(mheader2_t);
    }
  else
    {
      ret = DMI_read((int64_t)&(((dmi_memalloc_t*)dmi_memalloc_addr)->head_mheader2_addr_addr), sizeof(int64_t), &head_mheader2_addr, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      prev_mheader2_addr = head_mheader2_addr;
      ret = DMI_read(prev_mheader2_addr, sizeof(mheader2_t), &prev_mheader2, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      mheader2_addr = prev_mheader2.next_mheader2_addr;
      ret = DMI_read(mheader2_addr, sizeof(mheader2_t), &mheader2, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      while(1)
        {
          if(mheader2.size >= malloc_size)
            {
              if(mheader2.size == malloc_size)
                {
                  prev_mheader2.next_mheader2_addr = mheader2.next_mheader2_addr;
                  ret = DMI_write(prev_mheader2_addr, sizeof(mheader2_t), &prev_mheader2, DMI_PUT, NULL);
                  throw_or_catch(_wdmi->config->catch_flag, ret);
                }
              else
                {
                  mheader2.size -= malloc_size;
                  ret = DMI_write(mheader2_addr, sizeof(mheader2_t), &mheader2, DMI_PUT, NULL);
                  throw_or_catch(_wdmi->config->catch_flag, ret);
                  mheader2_addr += mheader2.size;
                  mheader2.size = malloc_size;
                  ret = DMI_write(mheader2_addr, sizeof(mheader2_t), &mheader2, DMI_PUT, NULL);
                  throw_or_catch(_wdmi->config->catch_flag, ret);
                }
              
              ret = DMI_write((int64_t)&(((dmi_memalloc_t*)dmi_memalloc_addr)->head_mheader2_addr_addr), sizeof(int64_t), &prev_mheader2_addr, DMI_PUT, NULL);
              throw_or_catch(_wdmi->config->catch_flag, ret);
              
              *addr_ptr = mheader2_addr + sizeof(mheader2_t);
              break;
            }
          if(mheader2_addr == head_mheader2_addr)
            {
              if(malloc_size < expand_size)
                {
                  mmap_size = expand_size;
                  //expand_size *= 2;
                }
              else
                {
                  mmap_size = malloc_size;
                }
              if(page_size <= 0)
                {
                  page_size = 4096;
                }
              page_num = mmap_size / page_size + 1;
              
              ret = DMI_mmap(&mheader2_addr, page_size, page_num, NULL);
              throw_or_catch(_wdmi->config->catch_flag, ret);
              
              mheader2.size = page_size * page_num;
              ret = DMI_write(mheader2_addr, sizeof(mheader2_t), &mheader2, DMI_PUT, NULL);
              throw_or_catch(_wdmi->config->catch_flag, ret);
              
              ret = dmi_memalloc_free(dmi_memalloc_addr, mheader2_addr + sizeof(mheader2_t), TRUE);
              throw_or_catch(_wdmi->config->catch_flag, ret);
              
              ret = DMI_read((int64_t)&(((dmi_memalloc_t*)dmi_memalloc_addr)->head_mheader2_addr_addr), sizeof(int64_t), &head_mheader2_addr, DMI_GET, NULL);
              throw_or_catch(_wdmi->config->catch_flag, ret);
              mheader2_addr = head_mheader2_addr;
            }
          
          prev_mheader2_addr = mheader2_addr;
          ret = DMI_read(prev_mheader2_addr, sizeof(mheader2_t), &prev_mheader2, DMI_GET, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
          mheader2_addr = mheader2.next_mheader2_addr;
          ret = DMI_read(mheader2_addr, sizeof(mheader2_t), &mheader2, DMI_GET, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
        }
    }
  
  if(rec_flag == FALSE)
    {
      ret = DMI_mutex_unlock((int64_t)&(((dmi_memalloc_t*)dmi_memalloc_addr)->dmi_mutex));
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t dmi_memalloc_realloc(int64_t dmi_memalloc_addr, int64_t old_addr, int64_t *new_addr_ptr, int64_t size, int64_t page_size, int8_t rec_flag)
{
  mheader2_t free_mheader2;
  int8_t *buf;
  int32_t ret;
  int64_t copy_size, free_mheader2_addr;
  
  if(rec_flag == FALSE)
    {
      ret = DMI_mutex_lock((int64_t)&(((dmi_memalloc_t*)dmi_memalloc_addr)->dmi_mutex));
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  if(old_addr == DMI_NULL)
    {
      ret = dmi_memalloc_malloc(dmi_memalloc_addr, new_addr_ptr, size, page_size, TRUE);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  else if(size == 0)
    {
      ret = dmi_memalloc_free(dmi_memalloc_addr, old_addr, TRUE);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      *new_addr_ptr = DMI_NULL;
    }
  else
    {
      free_mheader2_addr = old_addr - sizeof(mheader2_t);
      ret = DMI_read(free_mheader2_addr, sizeof(mheader2_t), &free_mheader2, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      if(free_mheader2.size < 0)
        {
          free_mheader2.size = -free_mheader2.size;
        }
      free_mheader2.size -= sizeof(mheader_t);
      copy_size = size >= free_mheader2.size ? free_mheader2.size : size;
      
      ret = dmi_memalloc_malloc(dmi_memalloc_addr, new_addr_ptr, size, page_size, TRUE);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      buf = (int8_t*)my_malloc(copy_size);
      ret = DMI_read(old_addr, copy_size, buf, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      ret = DMI_write(*new_addr_ptr, copy_size, buf, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      my_free(buf);
      
      ret = dmi_memalloc_free(dmi_memalloc_addr, old_addr, TRUE);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  if(rec_flag == FALSE)
    {
      ret = DMI_mutex_unlock((int64_t)&(((dmi_memalloc_t*)dmi_memalloc_addr)->dmi_mutex));
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t dmi_memalloc_free(int64_t dmi_memalloc_addr, int64_t addr, int8_t rec_flag)
{
  mheader2_t mheader2, free_mheader2, next_mheader2;
  int32_t ret;
  int64_t mheader2_addr, free_mheader2_addr, next_mheader2_addr, head_mheader2_addr;
  
  if(addr == DMI_NULL)
    {
      return TRUE;
    }
  
  if(rec_flag == FALSE)
    {
      ret = DMI_mutex_lock((int64_t)&(((dmi_memalloc_t*)dmi_memalloc_addr)->dmi_mutex));
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  free_mheader2_addr = addr - sizeof(mheader2_t);
  ret = DMI_read(free_mheader2_addr, sizeof(mheader2_t), &free_mheader2, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  if(free_mheader2.size < 0)
    {
      ret = DMI_munmap(free_mheader2_addr, NULL);
      if(ret < 0) error();
    }
  else
    {
      ret = DMI_read((int64_t)&(((dmi_memalloc_t*)dmi_memalloc_addr)->head_mheader2_addr_addr), sizeof(int64_t), &head_mheader2_addr, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      mheader2_addr = head_mheader2_addr;
      ret = DMI_read(mheader2_addr, sizeof(mheader2_t), &mheader2, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      while(1)
        {
          if((mheader2_addr < free_mheader2_addr && free_mheader2_addr < mheader2.next_mheader2_addr)
             || (mheader2_addr >= mheader2.next_mheader2_addr
                 && (mheader2_addr < free_mheader2_addr || free_mheader2_addr < mheader2.next_mheader2_addr)))
            {
              break;
            }
          mheader2_addr = mheader2.next_mheader2_addr;
          ret = DMI_read(mheader2_addr, sizeof(mheader2_t), &mheader2, DMI_GET, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
        }
      
      next_mheader2_addr = mheader2.next_mheader2_addr;
      ret = DMI_read(next_mheader2_addr, sizeof(mheader2_t), &next_mheader2, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      if(free_mheader2_addr + free_mheader2.size == next_mheader2_addr)
        {
          free_mheader2.size += next_mheader2.size;
          free_mheader2.next_mheader2_addr = next_mheader2.next_mheader2_addr;
        }
      else
        {
          free_mheader2.next_mheader2_addr = next_mheader2_addr;
        }
      
      ret = DMI_write(free_mheader2_addr, sizeof(mheader2_t), &free_mheader2, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      if(mheader2_addr + mheader2.size == free_mheader2_addr)
        {
          mheader2.size += free_mheader2.size;
          mheader2.next_mheader2_addr = free_mheader2.next_mheader2_addr;
        }
      else
        {
          mheader2.next_mheader2_addr = free_mheader2_addr;
        }
      ret = DMI_write(mheader2_addr, sizeof(mheader2_t), &mheader2, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_write((int64_t)&(((dmi_memalloc_t*)dmi_memalloc_addr)->head_mheader2_addr_addr), sizeof(int64_t), &mheader2_addr, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  if(rec_flag == FALSE)
    {
      ret = DMI_mutex_unlock((int64_t)&(((dmi_memalloc_t*)dmi_memalloc_addr)->dmi_mutex));
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

/* forkkill */

int32_t dmi_forkkill_init(int64_t dmi_forkkill_addr)
{
  dmi_forkkill_t dmi_forkkill;
  forkkill_t forkkill;
  int32_t ret;
  
  ret = DMI_mmap(&dmi_forkkill.dmi_mutex_addr, sizeof(dmi_mutex_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&dmi_forkkill.dmi_cond_addr, sizeof(dmi_cond_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mmap(&dmi_forkkill.forkkill_addr, sizeof(forkkill_t), 1, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  forkkill.phase = 0;
  forkkill.exit_dmi_id = SYS_ID_UNDEF;
  ret = DMI_write(dmi_forkkill.forkkill_addr, sizeof(forkkill_t), &forkkill, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_init(dmi_forkkill.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_cond_init(dmi_forkkill.dmi_cond_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_write(dmi_forkkill_addr, sizeof(dmi_forkkill_t), &dmi_forkkill, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_forkkill_destroy(int64_t dmi_forkkill_addr)
{
  dmi_forkkill_t dmi_forkkill;
  int32_t ret;
  
  ret = DMI_read(dmi_forkkill_addr, sizeof(dmi_forkkill_t), &dmi_forkkill, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_cond_destroy(dmi_forkkill.dmi_cond_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_mutex_destroy(dmi_forkkill.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_munmap(dmi_forkkill.forkkill_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_forkkill.dmi_cond_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  ret = DMI_munmap(dmi_forkkill.dmi_mutex_addr, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_forkkill_consumepre(int64_t dmi_forkkill_addr, forkkill_t *forkkill_ptr, int8_t *exit_flag_ptr)
{
  dmi_forkkill_t dmi_forkkill;
  forkkill_t forkkill;
  int32_t ret, dmi_id;
  
  ret = DMI_rank(&dmi_id);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_read(dmi_forkkill_addr, sizeof(dmi_forkkill_t), &dmi_forkkill, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_lock(dmi_forkkill.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  while(1)
    {
      ret = DMI_read(dmi_forkkill.forkkill_addr, sizeof(forkkill_t), &forkkill, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      if(forkkill.exit_dmi_id == dmi_id)
        {
          forkkill.exit_dmi_id = SYS_ID_UNDEF;
          ret = DMI_write(dmi_forkkill.forkkill_addr, sizeof(forkkill_t), &forkkill, DMI_PUT, NULL);
          throw_or_catch(_wdmi->config->catch_flag, ret);
          
          ret = DMI_cond_broadcast(dmi_forkkill.dmi_cond_addr);
          throw_or_catch(_wdmi->config->catch_flag, ret);
          
          ret = DMI_mutex_unlock(dmi_forkkill.dmi_mutex_addr);
          throw_or_catch(_wdmi->config->catch_flag, ret);
          
          *exit_flag_ptr = TRUE;
          break;
        }
      else if(forkkill.phase == 1 && forkkill.dmi_id == dmi_id)
        {
          *forkkill_ptr = forkkill;
          *exit_flag_ptr = FALSE;
          break;
        }
      
      ret = DMI_cond_wait(dmi_forkkill.dmi_cond_addr, dmi_forkkill.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  return TRUE;
}

int32_t dmi_forkkill_consumepost(int64_t dmi_forkkill_addr, dmi_node_t dmi_node)
{
  dmi_forkkill_t dmi_forkkill;
  forkkill_t forkkill;
  int32_t ret;
  
  ret = DMI_read(dmi_forkkill_addr, sizeof(dmi_forkkill_t), &dmi_forkkill, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  forkkill.phase = 2;
  forkkill.dmi_id = dmi_node.dmi_id;
  forkkill.dmi_node = dmi_node;
  
  ret = DMI_write(dmi_forkkill.forkkill_addr, sizeof(forkkill_t), &forkkill, DMI_PUT, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_cond_broadcast(dmi_forkkill.dmi_cond_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_unlock(dmi_forkkill.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_forkkill_produce(int64_t dmi_forkkill_addr, int8_t command_type, int32_t dmi_id, dmi_node_t *dmi_node_ptr, char *option)
{
  dmi_forkkill_t dmi_forkkill;
  forkkill_t forkkill;
  int32_t ret;
  
  ret = DMI_read(dmi_forkkill_addr, sizeof(dmi_forkkill_t), &dmi_forkkill, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_lock(dmi_forkkill.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  while(1)
    {
      ret = DMI_read(dmi_forkkill.forkkill_addr, sizeof(forkkill_t), &forkkill, DMI_INVALIDATE, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      if(forkkill.phase == 0)
        {
          break;
        }
      
      ret = DMI_cond_wait(dmi_forkkill.dmi_cond_addr, dmi_forkkill.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  forkkill.phase = 1;
  forkkill.command_type = command_type;
  forkkill.dmi_id = dmi_id;
  if(option != NULL)
    {
      memcpy(forkkill.option, option, strlen(option) + 1);
    }
  else
    {
      forkkill.option[0] = 0;
    }
  
  ret = DMI_write(dmi_forkkill.forkkill_addr, sizeof(forkkill_t), &forkkill, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_cond_broadcast(dmi_forkkill.dmi_cond_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  while(1)
    {
      ret = DMI_read(dmi_forkkill.forkkill_addr, sizeof(forkkill_t), &forkkill, DMI_INVALIDATE, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      if(forkkill.phase == 2)
        {
          break;
        }
      
      ret = DMI_cond_wait(dmi_forkkill.dmi_cond_addr, dmi_forkkill.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
    }
  
  if(dmi_node_ptr != NULL)
    {
      *dmi_node_ptr = forkkill.dmi_node;
    }
  
  forkkill.phase = 0;
  
  ret = DMI_write(dmi_forkkill.forkkill_addr, sizeof(forkkill_t), &forkkill, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_cond_broadcast(dmi_forkkill.dmi_cond_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  ret = DMI_mutex_unlock(dmi_forkkill.dmi_mutex_addr);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

/* profile */

int32_t dmi_profile_init(int64_t dmi_profile_addr, int8_t type, char *name)
{
  dmi_profile_t dmi_profile;
  idid_t *idids;
  FILE *fp;
  char *fname;
  int32_t ret, vm_id, num, fname_size, mthread_id;
  int32_t *thread_seqs;
  int64_t *vm_seqs;
  
  dmi_profile.type = type;
  
  switch(dmi_profile.type)
    {
    case PROFILE_NONE:
      break;
    case PROFILE_OUT:
      dmi_profile.vm_max = CONST_VM_MAX2;
      
      vm_seqs = (int64_t*)my_malloc(sizeof(int64_t) * dmi_profile.vm_max);
      
      ret = DMI_mmap(&dmi_profile.vm_seqs_addr, sizeof(int64_t) * dmi_profile.vm_max, 1, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      for(vm_id = 0; vm_id < dmi_profile.vm_max; vm_id++)
        {
          vm_seqs[vm_id] = SYS_ID_UNDEF;
        }
      ret = DMI_write(dmi_profile.vm_seqs_addr, sizeof(int64_t) * dmi_profile.vm_max, vm_seqs, DMI_EXCLUSIVE, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      my_free(vm_seqs);
      
      thread_seqs = (int32_t*)my_malloc(sizeof(int32_t) * CONST_MTHREAD_MAX);
      
      ret = DMI_mmap(&dmi_profile.thread_seqs_addr, sizeof(int32_t) * CONST_MTHREAD_MAX, 1, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      for(mthread_id = 0; mthread_id < CONST_MTHREAD_MAX; mthread_id++)
        {
          thread_seqs[mthread_id] = 0;
        }
      
      ret = DMI_write(dmi_profile.thread_seqs_addr, sizeof(int32_t) * CONST_MTHREAD_MAX, thread_seqs, DMI_EXCLUSIVE, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      my_free(thread_seqs);
      
      dmi_profile.name_size = strlen(name);
            
      ret = DMI_mmap(&dmi_profile.name_addr, dmi_profile.name_size, 1, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      ret = DMI_mmap(&dmi_profile.dmi_mutex_addr, sizeof(dmi_mutex_t), 1, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_write(dmi_profile.name_addr, strlen(name), name, DMI_EXCLUSIVE, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_mutex_init(dmi_profile.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    case PROFILE_IN:
      fname_size = strlen(name) + 10;
      fname = (char*)my_malloc(fname_size);
      
      snprintf(fname, fname_size, "%s.prof", name);
      
      fp = fopen(fname, "rb");
      if(fp == NULL) error();
      
      num = 1;
      ret = fread(&dmi_profile.vm_num, sizeof(int32_t), num, fp);
      if(ret != num) error();
      
      idids = (idid_t*)my_malloc(sizeof(idid_t) * dmi_profile.vm_num);
      
      num = dmi_profile.vm_num;
      ret = fread(idids, sizeof(idid_t), num, fp);
      if(ret != num) error();
            
      ret = DMI_mmap(&dmi_profile.idids_addr, sizeof(idid_t) * dmi_profile.vm_num, 1, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_write(dmi_profile.idids_addr, sizeof(idid_t) * dmi_profile.vm_num, idids, DMI_EXCLUSIVE, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      my_free(idids);
      fclose(fp);
      
      my_free(fname);
      
      thread_seqs = (int32_t*)my_malloc(sizeof(int32_t) * CONST_MTHREAD_MAX);
      
      ret = DMI_mmap(&dmi_profile.thread_seqs_addr, sizeof(int32_t) * CONST_MTHREAD_MAX, 1, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      for(mthread_id = 0; mthread_id < CONST_MTHREAD_MAX; mthread_id++)
        {
          thread_seqs[mthread_id] = 0;
        }
      
      ret = DMI_write(dmi_profile.thread_seqs_addr, sizeof(int32_t) * CONST_MTHREAD_MAX, thread_seqs, DMI_EXCLUSIVE, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      my_free(thread_seqs);
      
      ret = DMI_mmap(&dmi_profile.dmi_mutex_addr, sizeof(dmi_mutex_t), 1, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_mutex_init(dmi_profile.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    default:
      error();
    }
  
  ret = DMI_write(dmi_profile_addr, sizeof(dmi_profile_t), &dmi_profile, DMI_EXCLUSIVE, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  return TRUE;
}

int32_t dmi_profile_destroy(int64_t dmi_profile_addr)
{
  dmi_profile_t dmi_profile;
  int32_t ret;
  
  ret = DMI_read(dmi_profile_addr, sizeof(dmi_profile_t), &dmi_profile, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  switch(dmi_profile.type)
    {
    case PROFILE_NONE:
      break;
    case PROFILE_OUT:
      ret = DMI_mutex_destroy(dmi_profile.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_munmap(dmi_profile.dmi_mutex_addr, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      ret = DMI_munmap(dmi_profile.name_addr, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      ret = DMI_munmap(dmi_profile.thread_seqs_addr, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      ret = DMI_munmap(dmi_profile.vm_seqs_addr, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    case PROFILE_IN:
      ret = DMI_mutex_destroy(dmi_profile.dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_munmap(dmi_profile.thread_seqs_addr, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      ret = DMI_munmap(dmi_profile.dmi_mutex_addr, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      ret = DMI_munmap(dmi_profile.idids_addr, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    default:
      error();
    }
  return TRUE;
}

profile_t* profile_alloc(void)
{
  profile_t *profile;
  
  profile = (profile_t*)my_malloc(sizeof(profile_t));
  return profile;
}

void profile_free(profile_t *profile)
{
  my_free(profile);
  return;
}

int32_t profile_init(profile_t *profile, int64_t dmi_profile_addr)
{
  dmi_profile_t dmi_profile;
  char hostname[FNAME_SIZE];
  char *name, *fname;
  pid_t pid;
  int32_t ret, fname_size, num;
  int64_t packet_size;
  
  ret = DMI_read(dmi_profile_addr, sizeof(dmi_profile_t), &dmi_profile, DMI_GET, NULL);
  throw_or_catch(_wdmi->config->catch_flag, ret);
  
  profile->type = dmi_profile.type;
  _tls_profile_rec_flag = FALSE;
  pthread_mutex_init(&profile->mutex, NULL);
  
  switch(profile->type)
    {
    case PROFILE_NONE:
      break;
    case PROFILE_OUT:
      profile->vm_max = dmi_profile.vm_max;
      profile->vm_seqs_addr = dmi_profile.vm_seqs_addr;
      profile->thread_seqs_addr = dmi_profile.thread_seqs_addr;
      profile->dmi_mutex_addr = dmi_profile.dmi_mutex_addr;
      profile->record_vector = vector_alloc(CONTAINER_DEFAULT, CONTAINER_DEFAULT, NULL);
      
      name = (char*)my_malloc(dmi_profile.name_size + 1);
      
      ret = DMI_read(dmi_profile.name_addr, dmi_profile.name_size, name, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      name[dmi_profile.name_size] = 0;
      
      ret = gethostname(hostname, IP_SIZE);
      if(ret < 0) error();
      
      pid = getpid();
      if(pid < 0) error();
      
      fname_size = dmi_profile.name_size + FNAME_SIZE * 2;
      fname = (char*)my_malloc(fname_size);
      
      snprintf(fname, fname_size, "%s_%s_%d.dat", name, hostname, pid);
      
      profile->fp = fopen(fname, "wb");
      if(profile->fp == NULL) error();
      
      num = 1;
      packet_size = sizeof(packet_t);
      ret = fwrite(&packet_size, sizeof(int32_t), num, profile->fp);
      if(ret != num) error();
      
      my_free(fname);
      my_free(name);
      break;
    case PROFILE_IN:
      profile->vm_num = dmi_profile.vm_num;
      profile->dmi_mutex_addr = dmi_profile.dmi_mutex_addr;
      profile->thread_seqs_addr = dmi_profile.thread_seqs_addr;
      
      profile->idids = (idid_t*)my_malloc(sizeof(idid_t) * profile->vm_num);
      
      ret = DMI_read(dmi_profile.idids_addr, sizeof(idid_t) * dmi_profile.vm_num, profile->idids, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      break;
    default:
      error();
    }
  return TRUE;
}

int32_t profile_destroy(profile_t *profile)
{
  record_t *record;
  int32_t ret, num;
  
  switch(profile->type)
    {
    case PROFILE_NONE:
      break;
    case PROFILE_OUT:
      while(!vector_isempty(profile->record_vector))
        {
          record = vector_pop(profile->record_vector);
          
          num = 1;
          ret = fwrite(record, sizeof(record_t), num, profile->fp);
          if(ret != num) error();
          
          record_free(record);
        }
      
      vector_free(profile->record_vector);
      
      fclose(profile->fp);
      break;
    case PROFILE_IN:
      my_free(profile->idids);
      break;
    default:
      error();
    }
  
  pthread_mutex_destroy(&profile->mutex);
  return TRUE;
}

int32_t profile_record_mmap(profile_t *profile, int64_t dmi_addr)
{
  wdmi_thread_t *wdmi_thread;
  int32_t vm_id, thread_seq, ret, tmp_thread_seq;
  int64_t vm_seq;
  
  if(_tls_profile_rec_flag == FALSE)
    {
      _tls_profile_rec_flag = TRUE;
      
      wdmi_thread = _tls_wdmi_thread;
      if(wdmi_thread == NULL) error();
      
      vm_id = SYS_ADDR_VM_PART(dmi_addr);
      if(vm_id >= profile->vm_max) return FALSE;
      
      ret = DMI_mutex_lock(profile->dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_read(profile->thread_seqs_addr + wdmi_thread->mthread_id * sizeof(int32_t), sizeof(int32_t), &thread_seq, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      tmp_thread_seq = thread_seq++;
      
      ret = DMI_write(profile->thread_seqs_addr + wdmi_thread->mthread_id * sizeof(int32_t), sizeof(int32_t), &thread_seq, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_mutex_unlock(profile->dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      vm_seq = ((int64_t)wdmi_thread->mthread_id << 32ULL) | tmp_thread_seq;
      ret = DMI_write(profile->vm_seqs_addr + vm_id * sizeof(int64_t), sizeof(int64_t), &vm_seq, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      _tls_profile_rec_flag = FALSE;
    }
  return TRUE;
}

int32_t profile_record_munmap(profile_t *profile, int64_t dmi_addr)
{
  int32_t vm_id, i_value, ret;
  
  if(_tls_profile_rec_flag == FALSE)
    {
      _tls_profile_rec_flag = TRUE;
      
      vm_id = SYS_ADDR_VM_PART(dmi_addr);
      
      if(vm_id >= profile->vm_max) return FALSE;
      
      i_value = SYS_ID_UNDEF;
      ret = DMI_write(profile->vm_seqs_addr + vm_id * sizeof(int32_t), sizeof(int32_t), &i_value, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      _tls_profile_rec_flag = FALSE;
    }
  return TRUE;
}

int32_t profile_record_access(profile_t *profile, int64_t dmi_addr, int64_t size)
{
  record_t *record;
  int32_t vm_id, ret, record_num, record_id, num;
  int64_t page_offset, vm_seq;
  
  if(_tls_profile_rec_flag == FALSE)
    {
      _tls_profile_rec_flag = TRUE;
      
      vm_id = SYS_ADDR_VM_PART(dmi_addr);
      page_offset = SYS_ADDR_OFFSET_PART(dmi_addr);
      if(vm_id >= profile->vm_max) return FALSE;
      
      ret = DMI_read(profile->vm_seqs_addr + vm_id * sizeof(int64_t), sizeof(int64_t), &vm_seq, DMI_UPDATE, NULL);
      catch(ret);
      
      if(vm_seq != SYS_ID_UNDEF)
        {
          pthread_mutex_lock(&profile->mutex);
          
          record_num = vector_size(profile->record_vector);
          for(record_id = 0; record_id < record_num; record_id++)
            {
              record = vector_at(profile->record_vector, record_id);
              if(record->vm_seq == vm_seq && record->page_offset == page_offset && record->size == size)
                {
                  record->count++;
                  break;
                }
            }
          if(record_id == record_num)
            {
              if(record_num >= CONST_RECORD_MAX)
                {
                  while(!vector_isempty(profile->record_vector))
                    {
                      record = vector_pop(profile->record_vector);
                      
                      num = 1;
                      ret = fwrite(record, sizeof(record_t), num, profile->fp);
                      if(ret != num) error();
                      
                      record_free(record);
                    }
                }
              
              record = record_alloc();
              record->vm_seq = vm_seq;
              record->page_offset = page_offset;
              record->size = size;
              record->count = 1;
              
              vector_push(profile->record_vector, record);
            }
        }
      
      pthread_mutex_unlock(&profile->mutex);
      
      _tls_profile_rec_flag = FALSE;
    }
  return TRUE;
}

int32_t profile_decide_page_size(profile_t *profile, int64_t *page_size_ptr)
{
  wdmi_thread_t *wdmi_thread;
  int32_t ret, thread_seq, tmp_thread_seq, i;
  int64_t vm_seq;
  
  if(_tls_profile_rec_flag == FALSE)
    {
      _tls_profile_rec_flag = TRUE;
      
      wdmi_thread = _tls_wdmi_thread;
      if(wdmi_thread == NULL) error();
      
      ret = DMI_mutex_lock(profile->dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_read(profile->thread_seqs_addr + wdmi_thread->mthread_id * sizeof(int32_t), sizeof(int32_t), &thread_seq, DMI_GET, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      tmp_thread_seq = thread_seq++;
      
      ret = DMI_write(profile->thread_seqs_addr + wdmi_thread->mthread_id * sizeof(int32_t), sizeof(int32_t), &thread_seq, DMI_PUT, NULL);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      ret = DMI_mutex_unlock(profile->dmi_mutex_addr);
      throw_or_catch(_wdmi->config->catch_flag, ret);
      
      vm_seq = ((int64_t)wdmi_thread->mthread_id << 32ULL) | tmp_thread_seq;
      *page_size_ptr = 0;
      for(i = 0; i < profile->vm_num; i++)
        {
          if(profile->idids[i].id1 == vm_seq)
            {
              *page_size_ptr = profile->idids[i].id2;
              break;
            }
        }
      
      _tls_profile_rec_flag = FALSE;
    }
  return TRUE;
}

int8_t profile_get_type(profile_t *profile)
{
  return profile->type;
}

/* tools */

wdmi_t* wdmi_alloc(void)
{
  wdmi_t *wdmi;
  
  wdmi = (wdmi_t*)my_malloc(sizeof(wdmi_t));
  wdmi->profile = (profile_t*)my_malloc(sizeof(profile_t));
  return wdmi;
}

void wdmi_free(wdmi_t *wdmi)
{
  my_free(wdmi->profile);
  my_free(wdmi);
  return;
}

wdmi_thread_t* wdmi_thread_alloc(void)
{
  wdmi_thread_t *wdmi_thread;
  
  wdmi_thread = (wdmi_thread_t*)my_malloc(sizeof(wdmi_thread_t));
  wdmi_thread->dummy_mheader = mheader_alloc();
  return wdmi_thread;
}

void wdmi_thread_free(wdmi_thread_t *wdmi_thread)
{
  mheader_free(wdmi_thread->dummy_mheader);
  my_free(wdmi_thread);
  return;
}

scale_thread_t* scale_thread_alloc(void)
{
  scale_thread_t *scale_thread;
  
  scale_thread = (scale_thread_t*)my_malloc(sizeof(scale_thread_t));
  return scale_thread;
}

void scale_thread_free(scale_thread_t *scale_thread)
{
  my_free(scale_thread);
  return;
}

ticket_t* ticket_alloc(int64_t dmi_forkkill_addr)
{
  ticket_t *ticket;
  int32_t dmi_id, ret;
  
  ret = DMI_rank(&dmi_id);
  catch(ret);
  
  ticket = (ticket_t*)my_malloc(sizeof(ticket_t));
  
  ticket->option = NULL;
  ticket->command_type = COMMAND_NONE;
  ticket->dmi_forkkill_addr = dmi_forkkill_addr;
  ticket->pid_vector = vector_alloc(CONTAINER_DEFAULT, CONTAINER_DEFAULT, NULL);
  
  pthread_cond_init(&ticket->cond, NULL);
  pthread_mutex_init(&ticket->mutex, NULL);
  
  pthread_create(&ticket->pthread, NULL, ticket_proxyer1, ticket);
  
  ret = DMI_create2(&ticket->dmi_thread, dmi_id, (int64_t)ticket, 0, TRUE, THREAD_PROXY, 0, 0, NULL);
  catch(ret);
  return ticket;
}

void ticket_free(ticket_t *ticket)
{
  dmi_forkkill_t dmi_forkkill;
  forkkill_t forkkill;
  int8_t command_type;
  int32_t ret, dmi_id;
  
  ret = DMI_rank(&dmi_id);
  catch(ret);
  
  command_type = COMMAND_EXIT;
  ret = my_write(_ticket_channel[PIPE_IN], &command_type, sizeof(int8_t));
  if(ret < 0) error();
  
  ret = DMI_read(ticket->dmi_forkkill_addr, sizeof(dmi_forkkill_t), &dmi_forkkill, DMI_GET, NULL);
  catch(ret);
  
  ret = DMI_mutex_lock(dmi_forkkill.dmi_mutex_addr);
  catch(ret);
  
  while(1)
    {
      ret = DMI_read(dmi_forkkill.forkkill_addr, sizeof(forkkill_t), &forkkill, DMI_GET, NULL);
      catch(ret);
      
      if(forkkill.exit_dmi_id == SYS_ID_UNDEF)
        {
          break;
        }
      
      ret = DMI_cond_wait(dmi_forkkill.dmi_cond_addr, dmi_forkkill.dmi_mutex_addr);
      catch(ret);
    }
  
  forkkill.exit_dmi_id = dmi_id;
  ret = DMI_write(dmi_forkkill.forkkill_addr, sizeof(forkkill_t), &forkkill, DMI_PUT, NULL);
  catch(ret);
  
  ret = DMI_cond_broadcast(dmi_forkkill.dmi_cond_addr);
  catch(ret);
  
  ret = DMI_mutex_unlock(dmi_forkkill.dmi_mutex_addr);
  catch(ret);
  
  ret = DMI_join(ticket->dmi_thread, NULL, NULL);
  catch(ret);
  
  pthread_join(ticket->pthread, NULL);
  
  pthread_mutex_destroy(&ticket->mutex);
  pthread_cond_destroy(&ticket->cond);
  
  vector_free(ticket->pid_vector);
  my_free(ticket->option);
  my_free(ticket);
  return;
}

void* ticket_proxyer1(void *arg)
{
  ticket_t *ticket;
  int8_t command_type;
  int32_t ret;
  
  ticket = (ticket_t*)arg;
  while(1)
    {
      ret = my_read(_ticket_channel[PIPE_OUT], &command_type, sizeof(int8_t));
      if(ret < 0) error();
      
      if(command_type == COMMAND_EXIT)
        {
          break;
        }
      else if(command_type == COMMAND_SIGINT)
        {
          pthread_mutex_lock(&ticket->mutex);
          
          ticket->command_type = command_type;
          pthread_cond_broadcast(&ticket->cond);
          
          pthread_mutex_unlock(&ticket->mutex);
        }
      else
        {
          error();
        }
    }
  return NULL;
}

void ticket_proxyer2(ticket_t *ticket)
{
  forkkill_t forkkill;
  int8_t exit_flag;
  int32_t ret;
  
  while(1)
    {
      ret = dmi_forkkill_consumepre(ticket->dmi_forkkill_addr, &forkkill, &exit_flag);
      catch(ret);
      
      if(exit_flag == TRUE)
        {
          break;
        }
      
      switch(forkkill.command_type)
        {
        case COMMAND_FORK:
          pthread_mutex_lock(&ticket->mutex);
          
          ticket->command_type = forkkill.command_type;
          ticket->option = (char*)my_realloc(ticket->option, sizeof(strlen(forkkill.option) + 1));
          memcpy(ticket->option, forkkill.option, sizeof(strlen(forkkill.option) + 1));
          
          pthread_cond_broadcast(&ticket->cond);
          
          pthread_mutex_unlock(&ticket->mutex);
          break;
        case COMMAND_KILL:
          pthread_mutex_lock(&ticket->mutex);
          
          ticket->command_type = forkkill.command_type;
          pthread_cond_broadcast(&ticket->cond);
          
          pthread_mutex_unlock(&ticket->mutex);
          break;
        default:
          error();
        }
    }
  return;
}

void ticket_monitor(ticket_t *ticket)
{
  char option_num_buf[16];
  int8_t exit_flag;
  int32_t ret, i, child_pid, option_num;
  
  pthread_mutex_lock(&ticket->mutex);
  
  exit_flag = FALSE;
  while(1)
    {
      switch(ticket->command_type)
        {
        case COMMAND_NONE:
          break;
        case COMMAND_FORK:
          child_pid = fork();
          if(child_pid < 0) error();
          if(child_pid == 0)
            {
              option_num = 0;
              for(i = 0; i < strlen(ticket->option); i++)
                {
                  if(ticket->option[i] == ' ')
                    {
                      option_num++;
                    }
                }
              sprintf(option_num_buf, "%d", option_num + 5);
              ret = execl(_wdmi->config->argv[0], _wdmi->config->argv[0], ticket->option, "-i", "127.0.0.1", "-l", "0", "-w", option_num_buf, NULL);
              if(ret < 0) error();
            }
          else
            {
              vector_push(ticket->pid_vector, (void*)(intptr_t)child_pid);
            }
          break;
        case COMMAND_SIGINT:
          exit_flag = TRUE;
          break;
        case COMMAND_KILL:
          _wdmi->config->detach_flag = TRUE;
          
          exit_flag = TRUE;
          break;
        default:
          error();
        }
      if(exit_flag == TRUE)
        {
          break;
        }
      
      ticket->command_type = COMMAND_NONE;
      
      pthread_cond_wait(&ticket->cond, &ticket->mutex);
    }
  
  pthread_mutex_unlock(&ticket->mutex);
  return;
}

void ticket_joinchild(ticket_t *ticket)
{
  int status;
  int32_t pid_num, pid_id, child_pid;
  
  pid_num = vector_size(ticket->pid_vector);
  for(pid_id = 0; pid_id < pid_num; pid_id++)
    {
      child_pid = (int32_t)(intptr_t)vector_at(ticket->pid_vector, pid_id);
      waitpid(child_pid, &status, 0);
    }
  vector_clear(ticket->pid_vector);
  return;
}

void signal_handler(int signum, siginfo_t *siginfo, void *ucontext)
{
  static int32_t kill_count = 0;
  int8_t command_type;
  int32_t ret;
  
  if(signum == SIGINT)
    {
      kill_count++;
      if(kill_count > CONST_KILL_COUNT)
        {
          exit(1);
        }
      
      command_type = COMMAND_SIGINT;
      ret = my_write(_ticket_channel[PIPE_IN], &command_type, sizeof(int8_t));
      if(ret < 0) error();
    }
  return;
}

config_t* config_alloc(int argc, char **argv)
{
  config_t *config;
  int32_t i, option_num;
  
  config = (config_t*)my_malloc(sizeof(config_t));
  
  config->ip = NULL;
  config->port = CONST_DEFAULT_PORT;
  config->listen_port = CONST_DEFAULT_PORT;
  config->core_num = CONST_DEFAULT_CORE_NUM;
  config->mem_size = CONST_DEFAULT_MEM_SIZE;
  config->profile_type = PROFILE_NONE;
  config->profile_name = NULL;
  config->catch_flag = FALSE;
  config->detach_flag = FALSE;
  
  option_num = atoi(argv[argc - 1]);
  for(i = argc - 1 - option_num; i < argc - 1; i++)
    {
      if(!strcmp(argv[i], "-i") && i < argc - 1)
        {
          i++;
          config->ip = (char*)my_realloc(config->ip, IP_SIZE);
          strncpy(config->ip, argv[i], IP_SIZE);
        }
      else if(!strcmp(argv[i], "-p") && i < argc - 1)
        {
          i++;
          config->port = atoi(argv[i]);
        }
      else if(!strcmp(argv[i], "-l") && i < argc - 1)
        {
          i++;
          config->listen_port = atoi(argv[i]);
        }
      else if(!strcmp(argv[i], "-s") && i < argc - 1)
        {
          i++;
          config->mem_size = atoll(argv[i]);
        }
      else if(!strcmp(argv[i], "-c") && i < argc - 1)
        {
          config->catch_flag = TRUE;
        }
      else if(!strcmp(argv[i], "-w") && i < argc - 1)
        {
          config->detach_flag = TRUE;
        }
      else if(!strcmp(argv[i], "-r") && i < argc - 1)
        {
          i++;
          config->profile_name = (char*)my_realloc(config->profile_name, FNAME_SIZE);
          strncpy(config->profile_name, argv[i], FNAME_SIZE);
          config->profile_type = PROFILE_OUT;
        }
      else if(!strcmp(argv[i], "-f") && i < argc - 1)
        {
          i++;
          config->profile_name = (char*)my_realloc(config->profile_name, FNAME_SIZE);
          strncpy(config->profile_name, argv[i], FNAME_SIZE);
          config->profile_type = PROFILE_IN;
        }
      else
        {
          error();
        }
    }
  
  config->argc = argc - option_num - 1;
  config->argv = argv;
  return config;
}

void config_free(config_t *config)
{
  my_free(config->profile_name);
  my_free(config->ip);
  my_free(config);
  return;
}

record_t* record_alloc(void)
{
  record_t *record;
  
  record = (record_t*)my_malloc(sizeof(record_t));
  return record;
}

void record_free(record_t *record)
{
  my_free(record);
  return;
}

mheader_t* mheader_alloc(void)
{
  mheader_t *mheader;
  
  mheader = (mheader_t*)my_malloc(sizeof(mheader_t));
  return mheader;
}

void mheader_free(mheader_t *mheader)
{
  my_free(mheader);
  return;
}

int64_t sub_convert_addr(int64_t stackarea_addr, int64_t addr1, int64_t addr2)
{
  int32_t d, v1, v2;
  
  v1 = addr1 & 0x3;
  v2 = addr2 & 0x3;
  
  d = 0;
  if(d == v1 || d == v2)
    {
      d = 1;
      if(d == v1 || d == v2)
        {
          d = 2;
        }
    }
  return stackarea_addr + d;
}

int64_t sub_revert_addr(int64_t stackarea_addr)
{
  return stackarea_addr - (stackarea_addr & 0x3);
}
